package com.twentyfouri.tvlauncher.setupwizard.widgets

import android.content.Context
import android.graphics.drawable.Drawable
import android.util.AttributeSet
import android.view.View
import android.widget.ImageView
import android.widget.LinearLayout
import androidx.annotation.IntRange
import androidx.core.graphics.drawable.DrawableCompat
import com.twentyfouri.tvlauncher.setupwizard.R
import java.lang.Integer.max
import kotlin.random.Random

class CaptchaView @JvmOverloads constructor(
    context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
): LinearLayout(context, attrs, defStyleAttr) {

    private val random = Random.Default
    var result = -1
        private set

    init {
        orientation = HORIZONTAL
        regenerate()
    }

    companion object {
        private const val FIRST_NUMBER_DIGITS_COUNT: Int = 2  // Used as maximum count for equations
        private const val USE_EQUATION: Boolean = true
    }

    fun regenerate() {
        removeAllViews()
        val firstNumberDigits = mutableListOf<Int>()
        firstNumberDigits.add(random.nextInt(1, 10))

        val firstNumberDigitCount = if (USE_EQUATION) {
            max(random.nextInt(FIRST_NUMBER_DIGITS_COUNT) + 1, 2)  // At least to digits is needed
        }
        else {
            FIRST_NUMBER_DIGITS_COUNT
        }
        for (i in 0 until firstNumberDigitCount - 1) {  // This will ensure 2-<FIRST_NUMBER_DIGITS_COUNT> digit number
            firstNumberDigits.add(random.nextInt(10))
        }

        var firstNumber = 0

        firstNumberDigits.forEach {
            addElement(Element.Number(it))
            firstNumber *= 10
            firstNumber += it
        }

        if (USE_EQUATION) {
            val shouldAdd = random.nextBoolean()
            addElement(if (shouldAdd) Element.Plus else Element.Minus)

            val secondNumber = random.nextInt(10)
            addElement(Element.Number(secondNumber))

            addElement(Element.Equal)

            result = if (shouldAdd) {
                firstNumber + secondNumber
            } else {
                firstNumber - secondNumber
            }
        }
        else {
            result = firstNumber
        }
    }

    private sealed class Element {
        data class Number(@IntRange(from = 0, to = 9) val number: Int): Element()
        object Plus: Element()
        object Minus: Element()
        object Equal: Element()
    }

    private fun addElement(element: Element) {
        val drawable = ImageView(context).apply {
            setImageDrawable(resources.getDrawable(getDrawableResource(element), context.theme))
        }
        addView(drawable)
    }

    @androidx.annotation.DrawableRes
    private fun getDrawableResource(element: Element) = when (element) {
        is Element.Number -> when (element.number) {
            0 -> R.drawable.captcha_zero
            1 -> R.drawable.captcha_one
            2 -> R.drawable.captcha_two
            3 -> R.drawable.captcha_three
            4 -> R.drawable.captcha_four
            5 -> R.drawable.captcha_five
            6 -> R.drawable.captcha_six
            7 -> R.drawable.captcha_seven
            8 -> R.drawable.captcha_eight
            9 -> R.drawable.captcha_nine
            else -> throw RuntimeException("It doesn't comply to given integer range!")
        }
        is Element.Plus -> R.drawable.captcha_plus
        is Element.Minus -> R.drawable.captcha_minus
        is Element.Equal -> R.drawable.captcha_equal
    }
}