package com.twentyfouri.searchproxy.resourcegen

import java.lang.IllegalArgumentException
import java.util.HashMap

/**
 * A collection of string resources.
 */
class StringResources {
    private data class StringResourcesItem(
        val name:String,
        val value:String,
        val translatable:Boolean?
    )

    private val map = HashMap<String, StringResourcesItem>()

    fun size() = map.size

    fun add(name: String, value: String, translatable: Boolean? = null): StringResources {
        map[name] = StringResourcesItem(name, value, translatable)
        return this
    }

    fun add(resources: StringResources): StringResources {
        resources.forEach { name, value, translatable ->
            add(name, value, translatable)
        }
        return this
    }

    operator fun get(name: String) = map[name]?.value
        ?: throw IllegalArgumentException("String resource \"${name}\" not found")

    fun getOrNull(name: String) = map[name]?.value

    fun format(name: String, vararg args: String):String {
        var result = get(name)

        for ((index, argVal) in args.withIndex()) {
            result = result.replace("%${index + 1}\$s", argVal)
        }

        return result
    }

    fun forEach(call: (name:String, value:String, translatable: Boolean?) -> Unit) {
        map.values.forEach { call.invoke(it.name, it.value, it.translatable) }
    }

    fun <T> map(transformer: (name:String, value:String, translatable: Boolean?) -> T) =
        map.values.map {
            transformer.invoke(it.name, it.value, it.translatable)
        }

    fun copy() = StringResources().add(this)

    override fun equals(other: Any?): Boolean {
        if (other === this) return true
        return (other is StringResources) && (other.map == this.map)
    }

    override fun hashCode(): Int = map.hashCode()
}