package com.twentyfouri.searchproxy.resourcegen

import kotlin.collections.ArrayList
import kotlin.collections.HashSet

/**
 * Utility methods for handling string resources across different
 * variants.
 */
object StringResourceVariants {
    /**
     *
     */
    fun forAllVariants(src: ResourceDirs, flavorDimensions: List<Dimension>, consumer: VariantConsumer) {
        // Generate all combinations
        val variants = getAllVariants(flavorDimensions)
        println("About to generate for ${variants.size} variants...")

        variants.forEach { variant ->
            combineStringsAllLanguages(src, variant, consumer)
        }
    }

    /**
     * For the given combination of flavors, combine the string resources
     * and run the given consumer against it.
     *
     * Additionally, do the same for each language.
     */
    private fun combineStringsAllLanguages(
        src: ResourceDirs,
        variant: Variant,
        consumer: VariantConsumer
    ) {
        val variantName = variant.getVariantName()
        println("Combining strings for ${variantName}...")

        // Combine default strings
        val defaultStrings = combineStrings(src, variant)
        consumer.invoke(defaultStrings, variant, "")

        // Now combine for each language
        val allLangs = getAllLanguages(src, variant)

        allLangs.forEach { lang ->
            val langStrings = combineStrings(src, variant, lang)
            if (langStrings != defaultStrings) {
                consumer.invoke(langStrings, variant, lang)
            }
        }
    }
    /**
     * For the given variant and language, combine the string resources.
     *
     * This only grabs resources from each single flavor plus the default.
     * That means it will grab from, for example, "entel" and "catchups",
     * but not "entelCatchups".
     *
     * This will also not take build types (debug/release) into account.
     */
    private fun combineStrings(src: ResourceDirs, variant: List<FlavorInfo>, lang: String = ""):StringResources {
        val strings = src.readStrings().copy()
        if (lang != "") {
            strings.add(src.readStrings(lang = lang))
        }

        variant.forEach { flavor ->
            strings.add(src.readStrings(flavor.name))
            if (lang != "") {
                strings.add(src.readStrings(flavor.name, lang))
            }
        }
        return strings
    }

    private fun getAllLanguages(src: ResourceDirs, variant: List<FlavorInfo>): Set<String> {
        val allLangs = HashSet<String>(src.getLanguages())
        variant.forEach { flavor ->
            allLangs.addAll(src.getLanguages(flavor.name))
        }
        return allLangs
    }

    private fun getAllVariants(dimensions: List<Dimension>): List<Variant> =
        dimensions.fold(listOf<Variant>()) { list: List<Variant>, dimension: List<FlavorInfo> ->
            val result = arrayListOf<List<FlavorInfo>>()

            dimension.forEach { flavor ->
                if (list.isEmpty()) {
                    result.add(listOf(flavor))
                } else {
                    list.forEach { combination ->
                        val flavorCombination = ArrayList(combination)
                        flavorCombination.add(flavor)
                        result.add(flavorCombination)
                    }
                }
            }

            result
        }
}