package com.twentyfouri.tvlauncher.common.utils

import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.net.TrafficStats
import android.net.wifi.WifiInfo
import android.net.wifi.WifiManager
import android.os.Debug
import android.text.format.DateFormat
import android.text.format.Formatter
import android.view.KeyEvent
import android.view.View
import android.widget.TextView
import androidx.core.content.getSystemService
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.lifecycleScope
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_KEY_EVENT_LOG
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_ERROR_LOG
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.GlobalScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.cancel
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.launchIn
import kotlinx.coroutines.flow.onEach
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.io.Closeable
import kotlin.time.Duration.Companion.seconds
import kotlin.time.ExperimentalTime
import timber.log.Timber

@OptIn(ExperimentalTime::class)
class OnScreenExtendedLogger(lifecycleOwner: LifecycleOwner) : OSELLog, DefaultLifecycleObserver, Closeable {
    private val scope = lifecycleOwner.lifecycleScope

    private var enabled = false

    private var currentPhrase = ""
    private var keyTimeout: Job? = null
    private val osels = mutableSetOf<OSELBase>()
    private var selectedOsel: OSELBase? = null
    private var textView: TextView? = null

    init {
        lifecycleOwner.lifecycle.addObserver(this)
    }

    fun setOutput(textView: TextView) {
        this.textView = textView
    }

    fun addOSEL(osel: OSELBase) {
        osels += osel
    }

    fun removeOSEL(osel: OSELBase) {
        if (osel == selectedOsel) selectedOsel = osels.firstOrNull()
        osels -= osel
    }

    fun keyEvent(event: KeyEvent?) {
        if (event == null) {
            resetListening()
            return
        }

        if (event.action != KeyEvent.ACTION_DOWN) return

        currentPhrase += event.keyCharacterMap.getDisplayLabel(event.keyCode)
        startTimeout()
    }

    private fun resetListening() {
        currentPhrase = ""
        keyTimeout?.cancel()
        keyTimeout = null
    }

    private fun startTimeout() {
        keyTimeout?.cancel()
        keyTimeout = scope.launch {
            delay(KEY_PRESS_TIMEOUT)
            checkPhrase()
            currentPhrase = ""
        }
    }

    private suspend fun checkPhrase() {
        if (ENABLE_PHRASE.equals(currentPhrase, true)) {
            enabled = !enabled
            if (enabled) {
                withContext(Dispatchers.Main) { textView?.visibility = View.VISIBLE }
                if (selectedOsel == null) selectedOsel = osels.firstOrNull()
                selectedOsel?.startLog()
            } else {
                withContext(Dispatchers.Main) { textView?.visibility = View.GONE }
                selectedOsel?.stopLog()
                internalLog("")
            }
            return
        }

        if (enabled) {
            osels.forEachIndexed { index, it ->
                val qphrase = "q$index"
                if (it.phrase.equals(currentPhrase, true) || qphrase.equals(currentPhrase, true)) {
                    selectedOsel?.stopLog()
                    selectedOsel = it
                    selectedOsel?.startLog()
                    return
                }
            }
        }
    }

    private suspend fun internalLog(message: String) {
        withContext(Dispatchers.Main) {
            textView?.text = message
        }
    }

    override fun close() {
        osels.forEach {
            it.close()
        }
    }

    override suspend fun log(message: String, caller: OSELBase) {
        if (caller === selectedOsel) internalLog(message)
    }

    override fun onCreate(owner: LifecycleOwner) {
        super.onCreate(owner)
        addOSEL(OSELInfo(this, osels))
    }

    override fun onStart(owner: LifecycleOwner) {
        scope.launch {
            selectedOsel?.startLog()
        }
        super.onStart(owner)
    }

    override fun onStop(owner: LifecycleOwner) {
        selectedOsel?.stopLog()
        super.onStop(owner)
    }

    override fun onDestroy(owner: LifecycleOwner) {
        close()
        super.onDestroy(owner)
    }

    private companion object {
        private val KEY_PRESS_TIMEOUT = 1.seconds
        private const val ENABLE_PHRASE = "iddqd"
    }
}

interface OSELLog {
    suspend fun log(message: String, caller: OSELBase)
}

abstract class OSELBase(val oselLog: OSELLog) : Closeable {
    protected val scope = CoroutineScope(CoroutineName("OSELBase"))
    abstract val phrase: String
    abstract val desc: String
    abstract suspend fun startLog()
    abstract fun stopLog()
    override fun close() {
        scope.cancel()
    }
}

class OSELInfo(oselLog: OSELLog, val osels: Collection<OSELBase>) : OSELBase(oselLog) {
    override val phrase = "qinfo"
    override val desc = "This INFO"
    override suspend fun startLog() {
        oselLog.log(
            buildString {
                appendLine("On Screen Extended Logger (OSEL)")
                appendLine("activate logger by writing its phrase (for example qinfo)")
                osels.forEach {
                    appendLine("${it.phrase} -> ${it.desc}")
                }
            },
            this
        )
    }

    override fun stopLog() {}
}

class OSELNetStats(oselLog: OSELLog, val context: Context) : OSELBase(oselLog) {
    override val phrase = "qnet"
    override val desc = "Network TX/RX statistics"
    private var job: Job? = null

    override suspend fun startLog() {
        job?.cancel()
        val pid = android.os.Process.myUid()
        var oldTx = TrafficStats.getUidTxBytes(pid)
        var oldRx = TrafficStats.getUidRxBytes(pid)
        val startTx = oldTx
        val startRx = oldRx
        val hist: ArrayList<Pair<Long, Long>> = arrayListOf()
        hist.add(Pair(0, 0))
        job = scope.launch {
            while (isActive) {
                val newTx = TrafficStats.getUidTxBytes(pid)
                val newRx = TrafficStats.getUidRxBytes(pid)
                val diffTx = newTx - oldTx
                val diffRx = newRx - oldRx
                val avgRx = hist.sumBy { it.second.toInt() } / hist.size
                val avgTx = hist.sumBy { it.first.toInt() } / hist.size
                val res = buildString {
                    appendLine(desc)
                    appendLine("CURRENT (10s average)")
                    appendLine(" - DOWNLOAD: ${Formatter.formatFileSize(context, avgRx.toLong())}")
                    appendLine(" - UPLOAD: ${Formatter.formatFileSize(context, avgTx.toLong())}")
                    appendLine("TOTAL FROM DEVICE START")
                    appendLine(" - DOWNLOAD: ${Formatter.formatFileSize(context, newRx)}")
                    appendLine(" - UPLOAD: ${Formatter.formatFileSize(context, newTx)}")
                    appendLine("TOTAL FROM LOG START")
                    appendLine(" - DOWNLOAD: ${Formatter.formatFileSize(context, newRx - startRx)}")
                    append(" - UPLOAD: ${Formatter.formatFileSize(context, newTx - startTx)}")
                }
                oselLog.log(res, this@OSELNetStats)
                oldTx = newTx
                oldRx = newRx
                hist.add(Pair(diffTx, diffRx))
                if (hist.size >= 10) hist.removeAt(0)
                delay(1.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
    }
}

class OSELAppInfo(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qapp"
    override val desc = "About launcher and setupwizard"

    override suspend fun startLog() {
        val launcherInfo = ReflectionUtils.getPackageInfo()
        val wizardInfo = ReflectionUtils.getPackageInfo(Flavor().deviceInfoConfig.wizardPackage)
        val res = buildString {
            launcherInfo?.let {
                appendLine(desc)
                appendLine("Launcher name: ${it.versionName}")
                appendLine("Launcher code: ${it.versionCode}")
                appendLine("Launcher package: ${it.packageName}")
                appendLine("Launcher last update: ${DateFormat.format("dd.MM.yyyy HH:mm", it.lastUpdateTime)}")
            } ?: appendLine("Obtaining launcher info failed")

            wizardInfo?.let {
                appendLine("Setupwizard name: ${it.versionName}")
                appendLine("Setupwizard code: ${it.versionCode}")
                appendLine("Setupwizard package: ${it.packageName}")
                append("Setupwizard last update: ${DateFormat.format("dd.MM.yyyy HH:mm", it.lastUpdateTime)}")
            } ?: append("Obtaining setupwizard info failed")
        }
        oselLog.log(res, this)
    }

    override fun stopLog() {}
}

//on Entel this generate some error log lines but seems not affecting anything
class OSELMemoryInfo(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qmem"
    override val desc = "Memory info"
    private var job: Job? = null

    //TODO
    //check android.app.ActivityManager#getProcessMemoryInfo(int[])

    override suspend fun startLog() {
        job?.cancel()
        val memInfo = Debug.MemoryInfo()
        job = scope.launch {
            while (isActive) {
                Debug.getMemoryInfo(memInfo)
                memInfo.memoryStats
                val res = buildString {
                    appendLine(desc)
                    appendLine("!! Displaying this information may adversely affect performance !!")
                    appendLine("TOTAL: ${memInfo.totalPrivateClean + memInfo.totalPrivateDirty} kB")
                    appendLine("NATIVE: ${memInfo.nativePrivateDirty} kB")
                    appendLine("DALVIK: ${memInfo.dalvikPrivateDirty} kB")
                    append("OTHER: ${memInfo.otherPrivateDirty} kB")
                }
                oselLog.log(res, this@OSELMemoryInfo)
                delay(1.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
    }
}

class OSELNetworkLAN(oselLog: OSELLog, val context: Context) : OSELBase(oselLog), InfoUpdated {
    override val phrase = "qlan"
    override val desc = "Network LAN info"
    private var job: Job? = null
    private var networkInfo: NetworkInfo? = null

    override suspend fun startLog() {
        networkInfo = NetworkInfo(context, this)
        job?.cancel()
        job = scope.launch {
            while (isActive) {
                networkInfo?.update(context)
                delay(10.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
        networkInfo = null
    }

    override suspend fun onInfoUpdated() {
        oselLog.log("$desc\n${networkInfo?.ethernet?.toString(context) ?: "getting info failed"}", this)
    }
}

class OSELNetworkWIFI(oselLog: OSELLog, val context: Context) : OSELBase(oselLog), InfoUpdated {
    override val phrase = "qwifi"
    override val desc = "Network WIFI info"
    private var job: Job? = null
    private var networkInfo: NetworkInfo? = null

    override suspend fun startLog() {
        networkInfo = NetworkInfo(context, this)
        job?.cancel()
        job = scope.launch {
            while (isActive) {
                networkInfo?.update(context)
                delay(10.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
        networkInfo = null
    }

    override suspend fun onInfoUpdated() {
        oselLog.log("$desc\n${networkInfo?.wifi?.toString(context) ?: "getting info failed"}", this)
    }
}

class OSELEPG(oselLog: OSELLog, private val provideEpgInfo: () -> String) : OSELBase(oselLog) {
    override val phrase = "qepg"
    override val desc = "EPG info"
    private var job: Job? = null

    override suspend fun startLog() {
        job?.cancel()
        job = scope.launch {
            while (isActive) {
                val res = buildString {
                    appendLine(desc)
                    append(provideEpgInfo())
                }
                oselLog.log(res, this@OSELEPG)
                delay(2.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
    }
}

class OSELCrash(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qcra"
    override val desc = "Last crash log"

    override suspend fun startLog() {
        val res = buildString {
            appendLine(desc)
            append(SharedPreferencesUtils.getAppCrashStackTrace())
        }
        oselLog.log(res, this)
    }

    override fun stopLog() {
    }
}

class OSELSession(oselLog: OSELLog, private val context: Context) : OSELBase(oselLog) {
    override val phrase = "qses"
    override val desc = "Session information"
    private var job: Job? = null

    override suspend fun startLog() {
        job?.cancel()
        job = scope.launch {
            while (isActive) {
                val res = buildString {
                    appendLine(desc)
                    append(Flavor().getSessionInfo(context))
                }
                oselLog.log(res, this@OSELSession)
                delay(10.seconds)
            }
        }
    }

    override fun stopLog() {
        job?.cancel()
    }
}

class OSELExceptions(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qexc"
    override val desc = "Exceptions"

    private val exceptions = MutableStateFlow(emptyList<Throwable>())
    private val tree = OSELTree()
    private var job: Job? = null

    override suspend fun startLog() {
        Timber.plant(tree)
        job?.cancel()
        job = exceptions.onEach {
            oselLog.log(it.joinToString("\n"), this)
        }.launchIn(scope)
    }

    override fun stopLog() {
        Timber.uproot(tree)
        job?.cancel()
        job = null
        exceptions.value = emptyList()
    }

    private inner class OSELTree: Timber.DebugTree() {
        override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
            if (t != null) {
                exceptions.value = exceptions.value.toMutableList().apply {
                    add(t)
                }
            }
        }
    }
}

class OSELConnectivity(oselLog: OSELLog, context: Context) : OSELBase(oselLog) {
    override val phrase = "qconn"
    override val desc = "Connectivity"

    private val connectivityManager = context.getSystemService<ConnectivityManager>()!!
    private val wifiManager = context.getSystemService<WifiManager>()!!

    private val defaultNetworkCallback = object: ConnectivityManager.NetworkCallback() {
        override fun onLost(network: Network) {
            Timber.tag(TAG).i("Network lost $network")
            super.onLost(network)
        }

        override fun onCapabilitiesChanged(network: Network, capabilities: NetworkCapabilities) {
            super.onCapabilitiesChanged(network, capabilities)
            if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)) {
                val wifiInfo = try {
                    (capabilities.transportInfo as WifiInfo)
                } catch (nsme: NoSuchMethodError) {
                    wifiManager.connectionInfo
                } catch (cce: ClassCastException) {
                    wifiManager.connectionInfo
                }
                Timber.tag(TAG).i("Wi-Fi @ ${wifiInfo.frequency} ${WifiInfo.FREQUENCY_UNITS}")
            }
            if (capabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)) {
                Timber.tag(TAG).i("Ethernet")
            }
        }
    }

    override suspend fun startLog() {
        connectivityManager.registerDefaultNetworkCallback(defaultNetworkCallback)
    }

    override fun stopLog() {
        connectivityManager.unregisterNetworkCallback(defaultNetworkCallback)
    }

    companion object {
        const val TAG = "Connectivity"
    }
}

class OSELKeylog(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qkey"
    override val desc = "Key log"

    private val keys = MutableStateFlow(emptyList<String>())
    private val tree = OSELTree()
    private var job: Job? = null

    override suspend fun startLog() {
        Timber.plant(tree)
        job?.cancel()
        job = keys.onEach {
            oselLog.log(it.joinToString("\n"), this)
        }.launchIn(scope)
    }

    override fun stopLog() {
        Timber.uproot(tree)
        job?.cancel()
        job = null
        keys.value = emptyList()
    }

    private inner class OSELTree : Timber.DebugTree() {
        override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
            if (tag == TAG_KEY_EVENT_LOG) {
                keys.value = keys.value.toMutableList().apply {
                    add(message)
                }
            }
        }
    }
}

class OSELPlayerErrorLog(oselLog: OSELLog) : OSELBase(oselLog) {
    override val phrase = "qplayer"
    override val desc = "Player error log"

    private val errors = MutableStateFlow(emptyList<String>())
    private val tree = OSELTree()
    private var job: Job? = null

    override suspend fun startLog() {
        Timber.plant(tree)
        job?.cancel()
        job = errors.onEach {
            oselLog.log(it.joinToString("\n"), this)
        }.launchIn(scope)
    }

    override fun stopLog() {
        Timber.uproot(tree)
        job?.cancel()
        job = null
        errors.value = emptyList()
    }

    private inner class OSELTree : Timber.DebugTree() {
        override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
            if (tag == TAG_PLAYER_ERROR_LOG) {
                errors.value = errors.value.toMutableList().apply {
                    add(message)
                }
            }
        }
    }
}