package com.twentyfouri.tvlauncher.common.utils

import android.text.format.DateFormat
import android.util.Log
import androidx.annotation.VisibleForTesting
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.channels.actor
import kotlinx.coroutines.channels.consumeEach
import timber.log.Timber.DebugTree
import java.io.*
import java.util.zip.ZipEntry
import java.util.zip.ZipInputStream
import java.util.zip.ZipOutputStream

/** Can be used for logging into size-limited zip file.
 *  Not suitable for frequent logging because the CPU load is too excessive.
 *  Can be used for less frequent logging i.e exceptions logging.
 *  */

@OptIn(ObsoleteCoroutinesApi::class, ExperimentalCoroutinesApi::class)
class ZipFileTree(file: File, maxSize: Int, private val tagSelection: List<String>? = null) : DebugTree(),
    CoroutineScope by CoroutineScope(Dispatchers.Default) {

    @VisibleForTesting
    internal var actorIsEmpty = true
    internal var loggingEnabled = false

    private val actor = actor<String>(capacity = Channel.UNLIMITED) {
        consumeEach { line ->
            val logLines = withContext(Dispatchers.IO) {
                ZipInputStream(FileInputStream(file)).use {
                    it.nextEntry
                    it.readBytes()
                }
            }.decodeToString().lines().toMutableList().apply {
                addAll(0, line.lines())
            }.dropLastWhile { it.isEmpty() }

            var zipEntry = ZipEntry(FILE_NAME)
            val bytes = ByteArrayOutputStream()
            ZipOutputStream(bytes).use { zip ->
                zip.putNextEntry(zipEntry)
                zip.write(logLines.joinToString(System.lineSeparator()).encodeToByteArray())
            }
            var i = 0
            while (zipEntry.compressedSize > maxSize) {
                bytes.reset()
                zipEntry = ZipEntry(FILE_NAME)
                i++
                ZipOutputStream(bytes).use { zip ->
                    zip.putNextEntry(zipEntry)
                    zip.write(logLines.dropLast(i).joinToString(System.lineSeparator()).encodeToByteArray())
                }
            }

            withContext(Dispatchers.IO) {
                FileOutputStream(file)
                    .use { file ->
                        file.write(bytes.toByteArray())
                    }
            }
            actorIsEmpty = isEmpty
        }
    }

    override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
        val logMessageFormatted = "${getDateNowFormatted()} ${mapPriority(priority)}/$tag: $message"
        tagSelection?.let { tags ->
           if (tags.any { tag == it }) {
               actor.trySend(logMessageFormatted)
           }
        } ?: actor.trySend(logMessageFormatted)
    }

    override fun isLoggable(tag: String?, priority: Int): Boolean {
        if (!loggingEnabled) return false
        return super.isLoggable(tag, priority)
    }

    private fun mapPriority(priority: Int): String {
        return when (priority) {
            Log.VERBOSE -> "V"
            Log.DEBUG -> "D"
            Log.INFO -> "I"
            Log.WARN -> "W"
            Log.ERROR -> "E"
            Log.ASSERT -> "A"
            else -> "$priority"
        }
    }

    private fun getDateNowFormatted() = DateFormat.format("yyyy-MM-dd HH:mm:ss", TimeProvider.nowMs())

    @VisibleForTesting
    internal companion object {
        @VisibleForTesting
        internal const val FILE_NAME = "logcat.txt"
    }
}