package com.twentyfouri.tvlauncher.common.utils

import android.os.Environment
import android.os.StatFs
import android.text.format.DateFormat
import android.util.Log
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import kotlinx.coroutines.*
import timber.log.Timber
import timber.log.Timber.DebugTree
import java.io.*
import java.util.zip.ZipEntry
import java.util.zip.ZipOutputStream

class TextFileTree(
    private val pathName: String,
    private val baseFileName: String,
    private val tagSelection: Iterable<String> = emptySet(),
    private val maxFileSize: Int,
    private val storageLimit: Int,
    private val requestToStop: () -> Unit
) : DebugTree() {

    private var file: File
    private var stopped = false //prevent unwanted logging when logging is about to stop

    init {
        val numberOfFiles = File(pathName).listFiles()
            ?.filter { it.name.contains(baseFileName) }
            ?.size
            ?: 0
        file = File("$pathName/$baseFileName$numberOfFiles.txt")
    }

    override fun log(priority: Int, tag: String?, message: String, t: Throwable?) {
        if(stopped) return
        if (storageAvailableMb() < storageLimit) {
            stopped = true
            writeLog(priority, tag, message, t)
            writeLog(Log.INFO, TAG_FILE_LOG, "OSEL ENDED because of low storage capacity. Available ${storageAvailableMb()}MB, minimum required ${storageLimit}MB", null)
            requestToStop()
        } else if (tag in tagSelection || t != null) {
            if (file.sizeInMb > maxFileSize) {
                processFilledFile(createNext = true)
            }
            writeLog(priority, tag, message, t)
        }
    }

    private fun writeLog(priority: Int, tag: String?, message: String, t: Throwable?) {
        val logMessageFormatted = "${getDateNowFormatted()} ${mapPriority(priority)}/$tag: $message"
        GlobalScope.launch(Dispatchers.IO) {
            try {
                BufferedWriter(FileWriter(file, true)).use { writer ->
                    writer.newLine()
                    writer.append(logMessageFormatted)
                    if (t != null) {
                        writer.newLine()
                        writer.append("$t")
                    }
                }
            } catch (e: FileNotFoundException) {
                Timber.tag(TAG_FILE_LOG).e("File not found! ${e.message}")
            }
        }
    }

    fun disableLogging() {
        processFilledFile(createNext = false)
    }

    //TODO: may need update depending on whether the files should be stored in internal/external storage
    private fun storageAvailableMb(): Long {
        val path: File = Environment.getDataDirectory()
        val stat = StatFs(path.path)
        return stat.availableBytes / (1024 * 1024)
    }

    //creates new text file and zips current.
    private fun processFilledFile(createNext: Boolean) {
        try {
            val numberOfFiles = File(pathName).listFiles()
                ?.filter { it.name.contains(baseFileName) }
                ?.size
                ?: 0

            val fullFile = file
            file = File("$pathName/$baseFileName$numberOfFiles.txt")
            file.exists().ifFalse {
                try {
                    file.createNewFile()
                } catch (e: IOException) {
                    Timber.tag(TAG_FILE_LOG).e("FAILED TO CREATE NEW FILE (${file.name})")
                    e.printStackTrace()
                }
            }
            val zipFile = File("$pathName/$baseFileName${numberOfFiles - 1}.zip")
            zipFile.exists().ifFalse {
                try {
                    zipFile.createNewFile()
                } catch (e: IOException) {
                    Timber.tag(TAG_FILE_LOG).e("FAILED TO CREATE NEW FILE (${zipFile.name})")
                    e.printStackTrace()
                }
            }
            zip(zipFile, listOf(fullFile))
            fullFile.delete()
            if (!createNext) file.delete()
        } catch (e: SecurityException) {
            Timber.tag(TAG_FILE_LOG).e("Error processing log file: ${e.stackTraceToString()}")
        }
    }

    private fun mapPriority(priority: Int): String {
        return when (priority) {
            Log.VERBOSE -> "V"
            Log.DEBUG -> "D"
            Log.INFO -> "I"
            Log.WARN -> "W"
            Log.ERROR -> "E"
            Log.ASSERT -> "A"
            else -> "$priority"
        }
    }

    private fun getDateNowFormatted() =
        DateFormat.format("yyyy-MM-dd HH:mm:ss", TimeProvider.nowMs())

    private fun zip(zipFile: File, files: List<File>) {
        ZipOutputStream(BufferedOutputStream(FileOutputStream(zipFile))).use { outStream ->
            zip(outStream, files)
        }
    }

    private fun zip(outStream: ZipOutputStream, files: List<File>) {
        files.forEach { file ->
            outStream.putNextEntry(ZipEntry(file.name))
            BufferedInputStream(FileInputStream(file)).use { inStream ->
                inStream.copyTo(outStream)
            }
        }
    }

    private val File.size get() = if (!exists()) 0.0 else length().toDouble()
    private val File.sizeInMb get() = size / (1024 * 1024)

    companion object {

        const val TAG_FILE_LOG = "OSEL_FILE"

        suspend fun isStorageWritable(pathName: String, fileName: String): Boolean = withContext(Dispatchers.IO) {
            val directory = File(pathName)
            directory.exists().ifFalse {
                try {
                    if(!directory.mkdirs()) {
                        Timber.tag(TAG_FILE_LOG).w("FAILED TO CREATE LOG DIRECTORY ($pathName)")
                        return@withContext false
                    }
                } catch (e: IOException) {
                    Timber.tag(TAG_FILE_LOG).w("FAILED TO CREATE LOG DIRECTORY ($pathName)")
                    e.printStackTrace()
                    return@withContext false
                }
            }
            val testFileName = "$fileName.test"
            val file = File(directory, testFileName)
            try {
                file.exists().ifFalse {
                    file.createNewFile()
                }
                file.delete()
            } catch (e: IOException) {
                Timber.tag(TAG_FILE_LOG).w("FAILED TO WRITE TEST FILE ($testFileName)")
                e.printStackTrace()
                return@withContext false
            }
            Timber.tag(TAG_FILE_LOG).i("STORAGE TEST PASSED")
            return@withContext true
        }
    }
}