package com.twentyfouri.tvlauncher.common.utils

import android.annotation.SuppressLint
import android.content.Context
import android.content.SharedPreferences
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.SemaphoreState
import java.lang.Exception
import android.text.format.DateFormat

object SharedPreferencesUtils {

    private const val SHARED_PREFERENCES_PLAYER = "player"
    private const val SHARED_PREFERENCES_AUDIO_NAME = "audio_name"
    private const val SHARED_PREFERENCES_AUDIO_LANG = "audio_lang"
    private const val SHARED_PREFERENCES_SUBTITLE_NAME = "subtitle_name"
    private const val SHARED_PREFERENCES_SUBTITLE_LANG = "subtitle_lang"
    private const val SHARED_PREFERENCES_CHANNEL = "channel"
    private const val SHARED_PREFERENCES_LAST_CHANNEL = "last_channel"
    private const val SHARED_PREFERENCES_SESSION = "savedSession"
    private const val SHARED_PREFERENCES_LAST_SHUTDOWN_TIME = "last_shutdown_time"
    private const val SHARED_PREFERENCES_APP_CRASH_MESSAGE = "app_crash_data"
    private const val SHARED_PREFERENCES_APP_CRASH_STACK = "app_crash_stack"
    private const val SHARED_PREFERENCES_NETWORK_ERROR = "network_error"

    private const val SHARED_PREFERENCES_ONNET = "onnet"
    private const val SHARED_PREFERENCES_USER_TYPE = "user_type" //ftth=1, ott=2, other/missing=0, unused dth=3
    private const val SHARED_PREFERENCES_CHECK_NETWORK_QUALITY = "check_network_quality_for_new_user"
    private const val SHARED_PREFERENCES_NETWORK_QUALITY_SEMAPHORE = "network_quality_semaphore"
    private const val SHARED_PREFERENCES_APPROVED_NETWORKS = "network_quality_approved_network"
    private const val SHARED_PREFERENCES_INITIAL_APPROVED_NETWORK_SAVED = "network_quality_initial_network_saved"
    private const val SHARED_PREFERENCES_CHECK_NETWORK_QUALITY_REASON = "check_network_quality_reason"


    private const val SHARED_PREFERENCES_MAIN = "main"
    private const val SHARED_PREFERENCES_RESTART_COUNTER = "restart_counter"
    private const val SHARED_PREFERENCES_FORCE_OFFLINE = "force_offline"
    private const val SHARED_PREFERENCES_FORCE_RESTART = "force_restart"
    private const val SHARED_PREFERENCES_DEFAULT_RECOMMENDED_APPS = "default_recommended_apps"

    //OSEL REMOTE LOGGING ENABLED/DISABLED
    private const val SHARED_PREFERENCES_REMOTE_LOGGING_ENABLED = "remote_logging_enabled"
    private const val SHARED_PREFERENCES_REMOTE_LOGGING_STOP_DATE_MS = "remote_logging_stop_date_ms"
    private const val SHARED_PREFERENCES_REMOTE_LOGGING_STORAGE_LIMIT = "remote_logging_storage_limit"

    private fun getSharedPreferences(group: String): SharedPreferences? {
        return ReflectionUtils.getApplicationContext()?.getSharedPreferences(group, Context.MODE_PRIVATE)
    }

    @SuppressLint("ApplySharedPref")
    fun putChannelNumber(channelNumber: Int) {
        if (channelNumber != -1) { // we need to keep last channel remembered
            getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.edit()?.putInt(SHARED_PREFERENCES_LAST_CHANNEL, channelNumber)?.commit() // we need immediate change
        }
        getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.edit()?.putInt(SHARED_PREFERENCES_CHANNEL, channelNumber)?.apply()
    }

    fun getChannelNumber(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.getInt(SHARED_PREFERENCES_CHANNEL, -1) ?: -1
    }

    fun getLastChannelNumber(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.getInt(
                SHARED_PREFERENCES_LAST_CHANNEL, -1) ?: -1
    }

    fun getAudioSelection(name: String?, language: String?): Pair<String?, String?> {
        getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.let {
            return Pair(
                    it.getString(SHARED_PREFERENCES_AUDIO_NAME, name), //name
                    it.getString(SHARED_PREFERENCES_AUDIO_LANG, language) //language
            )
        }
        return Pair(name, language)
    }

    fun getSubtitleSelection(name: String?, language: String?): Pair<String?, String?> {
        getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.let {
            return Pair(
                    it.getString(SHARED_PREFERENCES_SUBTITLE_NAME, name),
                    it.getString(SHARED_PREFERENCES_SUBTITLE_LANG, language)
            )
        }
        return Pair(name, language)
    }

    fun putAudioSelection(name: String?, language: String?) {
        getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.edit()
                ?.putString(SHARED_PREFERENCES_AUDIO_NAME, name)
                ?.putString(SHARED_PREFERENCES_AUDIO_LANG, language)
                ?.apply()
    }

    fun putSubtitleSelection(name: String?, language: String?) {
        getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.edit()
                ?.putString(SHARED_PREFERENCES_SUBTITLE_NAME, name)
                ?.putString(SHARED_PREFERENCES_SUBTITLE_LANG, language)
                ?.apply()
    }

    @SuppressLint("ApplySharedPref") //intentionally commit because restart of activity may be imminent
    fun putRestartCounter(counter: Int) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()
                ?.putInt(SHARED_PREFERENCES_RESTART_COUNTER, counter)
                ?.commit()
    }

    fun getRestartCounter(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getInt(SHARED_PREFERENCES_RESTART_COUNTER, 0) ?: 0
    }

    @SuppressLint("ApplySharedPref") //intentionally commit because restart of activity may be imminent
    fun putForceOffline(forceOffline: Boolean) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()
                ?.putBoolean(SHARED_PREFERENCES_FORCE_OFFLINE, forceOffline)
                ?.commit()
    }

    fun getForceOffline(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getBoolean(SHARED_PREFERENCES_FORCE_OFFLINE, false) ?: false
    }

    fun getSessionData(): String? {
        return getSharedPreferences(SHARED_PREFERENCES_SESSION)?.getString("data", "")
    }

    @SuppressLint("ApplySharedPref")
    fun putLastShutDownTime() {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putLong(
                SHARED_PREFERENCES_LAST_SHUTDOWN_TIME, System.currentTimeMillis()
        )?.commit() // we need immediate change
    }


    fun getLastShutDownTime(): Long {
        val data = getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getLong(SHARED_PREFERENCES_LAST_SHUTDOWN_TIME, -1L) ?: -1L
        if (data > -1L) getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putLong(SHARED_PREFERENCES_LAST_SHUTDOWN_TIME, -1L)?.apply()
        return data
    }

    @SuppressLint("ApplySharedPref")
    fun putAppCrashData(exception: Throwable) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(
                SHARED_PREFERENCES_APP_CRASH_MESSAGE, exception.message
        )?.commit() // we need immediate change
    }

    @SuppressLint("ApplySharedPref")
    fun putAppCrashStackTrace(exception: Throwable) {
        val now = DateFormat.format("dd.MM.yyyy HH:mm", TimeProvider.nowMs())
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(
                SHARED_PREFERENCES_APP_CRASH_STACK,  "$now ${exception.stackTraceToString()}"
        )?.commit() // we need immediate change
    }

    fun getAppCrashData(): String {
        val data = getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getString(SHARED_PREFERENCES_APP_CRASH_MESSAGE, "") ?: ""
        if (data.isNotEmpty()) getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(SHARED_PREFERENCES_APP_CRASH_MESSAGE, "")?.apply()
        return data
    }

    //this one is intentionally not cleared after read (different from getAppCrashData())
    fun getAppCrashStackTrace(): String {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getString(SHARED_PREFERENCES_APP_CRASH_STACK, "No crash stack trace") ?: "No crash stack trace"
    }

    fun putConnectivityErrorState(state: String) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(SHARED_PREFERENCES_NETWORK_ERROR, state)?.apply()
    }

    fun getConnectivityErrorState(): String {
        val data = getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getString(SHARED_PREFERENCES_NETWORK_ERROR, "") ?: ""
        if (data.isNotEmpty()) getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(SHARED_PREFERENCES_NETWORK_ERROR, "")?.apply()
        return data
    }

    fun getLastOnnet(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getInt(SHARED_PREFERENCES_ONNET, 0) ?: 0
    }

    fun putLastOnnet(onnet: Int) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putInt(SHARED_PREFERENCES_ONNET, onnet)?.apply()
    }

    fun getLastUserType(): String? {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getString(SHARED_PREFERENCES_USER_TYPE, null)
    }

    fun putLastUserType(userType: String) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(SHARED_PREFERENCES_USER_TYPE, userType)?.apply()
    }

    fun isFirstTimeDeviceUsage(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_PLAYER)?.all?.isEmpty() == true
    }
    @SuppressLint("ApplySharedPref")
    fun putShouldCheckNetworkQuality(shouldCheck: Boolean) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putBoolean(SHARED_PREFERENCES_CHECK_NETWORK_QUALITY, shouldCheck)?.commit()
    }

    fun getShouldCheckNetworkQuality(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getBoolean(SHARED_PREFERENCES_CHECK_NETWORK_QUALITY, false) ?: false
    }

    fun putConnectionSemaphoreState(state: SemaphoreState) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putString(SHARED_PREFERENCES_NETWORK_QUALITY_SEMAPHORE, state.name)?.apply()
    }

    fun getNetworkQualityCheckReason(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)
            ?.getInt(SHARED_PREFERENCES_CHECK_NETWORK_QUALITY_REASON, 0) ?: 0
    }

    fun putNetworkQualityCheckReason(reason: Int) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()
            ?.putInt(SHARED_PREFERENCES_CHECK_NETWORK_QUALITY_REASON, reason)
            ?.apply()
    }

    fun getConnectionSemaphoreState(): SemaphoreState {
        return try {
            SemaphoreState.valueOf(getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getString(SHARED_PREFERENCES_NETWORK_QUALITY_SEMAPHORE, "") ?: "")
        } catch (e: Exception) {
            SemaphoreState.OFF
        }
    }

    fun getForceRestart(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getBoolean(SHARED_PREFERENCES_FORCE_RESTART, false) ?: false
    }

    @SuppressLint("ApplySharedPref") //intentionally commit because restart of activity may be imminent
    fun putForceRestart(forceRestart: Boolean) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()
                ?.putBoolean(SHARED_PREFERENCES_FORCE_RESTART, forceRestart)
                ?.commit()
    }

    fun putDefaultRecommendedApps(apps: Set<String>?){
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putStringSet(
            SHARED_PREFERENCES_DEFAULT_RECOMMENDED_APPS, apps)?.commit()
    }

    fun getDefaultRecommendedApps(): Set<String>? {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getStringSet(
            SHARED_PREFERENCES_DEFAULT_RECOMMENDED_APPS, null)
    }

    fun getApprovedNetworks(): Set<String>? {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getStringSet(
            SHARED_PREFERENCES_APPROVED_NETWORKS,null)
    }

    fun putApprovedNetworks(networkIds: Set<String>?) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putStringSet(
            SHARED_PREFERENCES_APPROVED_NETWORKS, networkIds
        )?.commit()
    }

    fun getIsInitialUsersNetworkSaved(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getBoolean(
            SHARED_PREFERENCES_INITIAL_APPROVED_NETWORK_SAVED,
            false) ?: false
    }

    fun putIsInitialUsersNetworkSaved(isSaved: Boolean) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putBoolean(
            SHARED_PREFERENCES_INITIAL_APPROVED_NETWORK_SAVED, isSaved
        )?.commit()
    }

    fun putRemoteLoggingEnabled(loggingEnabled: Boolean) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putBoolean(
            SHARED_PREFERENCES_REMOTE_LOGGING_ENABLED, loggingEnabled
        )?.apply()
    }

    fun getRemoteLoggingEnabled(): Boolean {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getBoolean(
            SHARED_PREFERENCES_REMOTE_LOGGING_ENABLED, false) ?: false
    }

    fun putRemoteLoggingStopDate(stopDateMs: Long) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putLong(
            SHARED_PREFERENCES_REMOTE_LOGGING_STOP_DATE_MS, stopDateMs
        )?.apply()
    }

    fun getRemoteLoggingStopDate(): Long {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getLong(
            SHARED_PREFERENCES_REMOTE_LOGGING_STOP_DATE_MS, 0) ?: 0
    }

    fun putRemoteLoggingStorageLimit(storageLimitMB: Int) {
        getSharedPreferences(SHARED_PREFERENCES_MAIN)?.edit()?.putInt(
            SHARED_PREFERENCES_REMOTE_LOGGING_STORAGE_LIMIT, storageLimitMB
        )?.apply()
    }

    fun getRemoteLoggingStorageLimit(): Int {
        return getSharedPreferences(SHARED_PREFERENCES_MAIN)?.getInt(
            SHARED_PREFERENCES_REMOTE_LOGGING_STORAGE_LIMIT, 0) ?: 0
    }

}