package com.twentyfouri.tvlauncher.common.utils

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.view.KeyEvent
import android.view.View
import androidx.fragment.app.DialogFragment
import androidx.lifecycle.MutableLiveData
import com.google.firebase.remoteconfig.FirebaseRemoteConfig
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.error.AppCanRunWithLimitationsException
import com.twentyfouri.smartmodel.model.error.ConcurrencyLimitationException
import com.twentyfouri.smartmodel.model.error.UnsubscribedChannelException
import com.twentyfouri.smartmodel.model.media.SmartRestrictionType
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.ui.MainActivityAction
import com.twentyfouri.tvlauncher.common.ui.messagedialog.*
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogFragment
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.pindialog.PinDialogModel

object RestrictionChecker {

    var dialogFragment: DialogFragment? = null

    private var isParentalPinCached: Boolean = false
    private var limitationDialogShown: Boolean = false
    private var networkCapabilityEth: Boolean? = null
    private var networkCapabilityWifi: Boolean? = null
    private var networkCapabilityMobile: Boolean? = null
    private var networkCapabilityVPN: Boolean? = null

    /* Check restriction should take care of everything. It will either:
     call `doOnSuccess` (if there are no restrictions)
     or shows dialog (if restriction can be overriden - eg. PIN), then calls `doOnSuccess`
     or shows dialog and does nothing else

     there should be no reason to return anything or to do anything from outside
    */
    fun checkRestrictions(
        context: Context,
        mediaItem: SmartMediaItem?,
        isInPlayerMode: Boolean = false,
        doOnSuccess: () -> Unit = {},
        doOnRestrictionBlock: () -> Unit = {},
        goToPreviousChannel: () -> Unit = {},
        goToNextChannel: () -> Unit = {},
        goToLastPlayedChannel: () -> Unit = {},
        useButtonDialog: Boolean = true
    ) {
        if(mediaItem == null) {
            //no event so no way how to check parental here
            doOnSuccess.invoke()
            return
        }

        mediaItem.restrictions.forEach {restriction -> //at this point we still do not check if restriction is broken or not. It needs to be mapped properly in smartModel

            when (restriction.type) {
                SmartRestrictionType.PARENTAL_PIN -> {
                    if (!GlobalPinTimer.isRunning(context, TimeProvider.nowMs()) || !isParentalPinCached) {
                        doOnRestrictionBlock.invoke()
                        showPinDialog(
                                context = context,
                                doOnSuccess = {
                                    isParentalPinCached = true
                                    checkRestrictions(
                                    context,
                                    mediaItem,
                                    isInPlayerMode,
                                    doOnSuccess,
                                    doOnRestrictionBlock,
                                    goToPreviousChannel,
                                    goToNextChannel,
                                    goToLastPlayedChannel
                                )},
                                isInPlayerMode = isInPlayerMode,
                                goToPreviousChannel = goToPreviousChannel,
                                goToNextChannel = goToNextChannel,
                                goToLastPlayedChannel = goToLastPlayedChannel
                        )
                        return
                    }
                }
                SmartRestrictionType.PARENTAL_PIN_BLOCKED -> {
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.unexpected_error_common ,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return
                } // age restriction in effect, override by PIN (temporarily) blocked (too many tries)
                SmartRestrictionType.PARENTAL_BLOCKED -> {
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.unexpected_error_common ,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return
                } // age restriction in effect, not negotiable
                SmartRestrictionType.GEO -> {
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.unexpected_error_common ,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return
                } // not available in your region
                SmartRestrictionType.DEVICE -> {
                    //should play
                } // device not added to account
                SmartRestrictionType.ANONYMOUS -> {
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.unexpected_error_common ,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return
                } // anonymous access denied
                SmartRestrictionType.USER -> {
                    //should play
                } // user is in wrong state: suspended, insufficient level (free/premium)
                SmartRestrictionType.PARALLEL_LIMIT -> {
                    //should play will be handlel after getPlaybackContext is called
                } // too many streams played in parallel, stop playing on other devices
                SmartRestrictionType.DRM_INSECURE -> {
                    //should play
                }    // device is sharing its screen through insecure display, do not allow DRM playback
                SmartRestrictionType.MVPD -> {
                    //should play
                } // user hasn't subscription but could watch with MVPD provider
                SmartRestrictionType.DEVICE_OUT_OF_HOME -> {
                    (context as? MainActivityAction)?.reportApiException(AppCanRunWithLimitationsException())
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.out_of_home,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return
                } // device is out of user's home
                SmartRestrictionType.DEVICE_OUT_OF_NETWORK -> {
                    doOnRestrictionBlock.invoke()
                    showLimitationErrorDialog(
                        context,
                        R.string.out_of_network,
                        isInPlayerMode,
                        goToPreviousChannel,
                        goToNextChannel,
                        goToLastPlayedChannel
                    )
                    return

                } // device is out of required network
                SmartRestrictionType.RECORDING -> {
                    //should play
                } // recording not allowed
                SmartRestrictionType.LIVE_NOT_READY -> {
                    //should play
                }  // live stream is not running at the moment
                SmartRestrictionType.FAST_FORWARD -> {
                    //should play
                } //fast forward is not allowed
                SmartRestrictionType.NOT_RELEASED -> {
                    getNetworkCapability(context)
                    if (networkCapabilityMobile == true && restriction.name == Flavor().networkCapabilityMobile) {
                        doOnRestrictionBlock.invoke()
                        showNotReleasedLimitationErrorDialog(
                            context,
                            Flavor().messageLicenceMobile(context),
                            isInPlayerMode,
                            goToPreviousChannel,
                            goToNextChannel,
                            goToLastPlayedChannel,
                            useButtonDialog
                        )
                        return
                    }
                    if (networkCapabilityWifi == true && restriction.name == Flavor().networkCapabilityWifi) {
                        doOnRestrictionBlock.invoke()
                        showNotReleasedLimitationErrorDialog(
                            context,
                            Flavor().messageLicenceWiFi(context),
                            isInPlayerMode,
                            goToPreviousChannel,
                            goToNextChannel,
                            goToLastPlayedChannel,
                            useButtonDialog
                        )
                        return
                    }
                    if (networkCapabilityEth == true && restriction.name == Flavor().networkCapabilityEth) {
                        doOnRestrictionBlock.invoke()
                        showNotReleasedLimitationErrorDialog(
                            context,
                            Flavor().messageLicenceEth(context),
                            isInPlayerMode,
                            goToPreviousChannel,
                            goToNextChannel,
                            goToLastPlayedChannel,
                            useButtonDialog
                        )
                        return
                    }
                    if (networkCapabilityVPN == true) {
                        doOnRestrictionBlock.invoke()
                        showNotReleasedLimitationErrorDialog(
                            context,
                            context.getString(R.string.content_not_licensed),
                            isInPlayerMode,
                            goToPreviousChannel,
                            goToNextChannel,
                            goToLastPlayedChannel,
                            useButtonDialog
                        )
                        return
                    }
                }  // not released for playback (mostly because of licensing)
                else -> {}
            }
        }

        //no restriction detected, we can continue
        doOnSuccess.invoke()
    }

    private fun getNetworkCapability(context: Context) {
        if (networkCapabilityEth != null && networkCapabilityMobile != null && networkCapabilityWifi != null) return

        val connectivityManager = context.getSystemService(ConnectivityManager::class.java) ?: return
        val networkCapabilities = connectivityManager.getNetworkCapabilities(connectivityManager.activeNetwork) ?: return
        networkCapabilityEth = networkCapabilityEth
            ?: networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET)
        networkCapabilityWifi = networkCapabilityWifi
            ?: networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI)
        networkCapabilityMobile = networkCapabilityMobile
            ?: networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR)
        networkCapabilityVPN = networkCapabilityVPN
                ?: networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_VPN)
    }

    fun handlePlaybackRestrictions(
        exception: Exception,
        playFailedVisibility: MutableLiveData<Int>,
        playFailedText: MutableLiveData<String>,
        resourceRepository: ResourceRepository,
        subscriptionChannel: MutableLiveData<SmartMediaItem?>,
        playingChannel: SmartMediaItem?,
        isPlayingCatchup: Boolean,
        isCatchupSubscribed: MutableLiveData<Boolean>
    ) {
        when (exception) {
            is ConcurrencyLimitationException -> {
                playFailedVisibility.postValue(View.VISIBLE)
                playFailedText.postValue(resourceRepository.getString(R.string.play_failed_concurrency))
            }
            is UnsubscribedChannelException -> {
                if (!limitationDialogShown) {
                    subscriptionChannel.postValue(playingChannel)
                }
                if (Flavor().allowSubscriptionPurchaseScreen().not()) {
                    playFailedVisibility.postValue(View.VISIBLE)
                    playFailedText.postValue(
                        if (playingChannel == null) {
                            resourceRepository.getString(R.string.play_failed_channel)
                        } else {
                            resourceRepository.getString(R.string.play_failed_subscription)
                        }
                    )
                    if (isPlayingCatchup) {
                        isCatchupSubscribed.postValue(false)
                    }
                }
            }
            else -> { }
        }
    }

    private fun showPinDialog(
        context: Context,
        doOnSuccess: () -> Unit = {},
        isInPlayerMode: Boolean = false,
        goToPreviousChannel: () -> Unit = {},
        goToNextChannel: () -> Unit = {},
        goToLastPlayedChannel: () -> Unit = {}
    ) {
        val messageDialogModel = PinDialogModel(
            context.getString(R.string.dialog_parental_pin_message, context.resources.getInteger(R.integer.pin_length)),
            context.getString(R.string.dialog_parental_pin_description))
        dialogFragment = PinDialogFragment.newInstance(messageDialogModel)
        (dialogFragment as PinDialogFragment).mListener = object : PinDialogFragmentListener {
            override fun onResult(answer: Boolean?, keyCode: Int?) {
                if (answer == true) {
                    GlobalPinTimer.saveTime(context)
                    doOnSuccess.invoke()
                } else if (isInPlayerMode) {
                    when (keyCode) {
                        KeyEvent.KEYCODE_CHANNEL_UP -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToNextChannel.invoke()
                        }
                        KeyEvent.KEYCODE_CHANNEL_DOWN -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToPreviousChannel.invoke()
                        }
                        KeyEvent.KEYCODE_BACK -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToLastPlayedChannel.invoke()
                        }
                    }
                }
            }
        }
        dialogFragment?.let {
            NavigatorCommon.getInstance().navigateDialog(it)
        }
    }

    fun dismissRestrictionDialog() {
        limitationDialogShown = false
        if (dialogFragment?.isAdded == true) {
            dialogFragment?.dismissAllowingStateLoss()
            dialogFragment = null
        }
    }

    private fun showLimitationErrorDialog(
        context: Context,
        messageId: Int,
        //doOnSuccess: () -> Unit = {},
        isInPlayerMode: Boolean = false,
        goToPreviousChannel: () -> Unit = {},
        goToNextChannel: () -> Unit = {},
        goToLastPlayedChannel: () -> Unit = {}
    ) {
        limitationDialogShown = true
        val messageDialogModel = MessageDialogModel(
            context.getString(messageId),
            null,
            context.getString(R.string.button_back),
            MessageDialogCodes.appCannotRunRuntime
        )
        dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        (dialogFragment as MessageDialogFragment).mListener = object :
            MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                limitationDialogShown = false
                if (answer is MessageDialogAction.Event && isInPlayerMode) {
                    when (answer.keyEvent.keyCode) {
                        KeyEvent.KEYCODE_CHANNEL_UP -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToNextChannel.invoke()
                        }
                        KeyEvent.KEYCODE_CHANNEL_DOWN -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToPreviousChannel.invoke()
                        }
                        KeyEvent.KEYCODE_BACK -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToLastPlayedChannel.invoke()
                        }
                    }
                    return true
                }
                return false
            }
        }
        (dialogFragment as MessageDialogFragment).mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {
                limitationDialogShown = false
            }
        }
        dialogFragment?.let {
            NavigatorCommon.getInstance().navigateDialog(it)
        }
    }

    private fun showNotReleasedLimitationErrorDialog(
            context: Context,
            message: String,
            isInPlayerMode: Boolean = false,
            goToPreviousChannel: () -> Unit = {},
            goToNextChannel: () -> Unit = {},
            goToLastPlayedChannel: () -> Unit = {},
            useButtonDialog: Boolean
    ) {
        limitationDialogShown = true
        val optionButtons = if (useButtonDialog) arrayOf<String?>((context.getString(R.string.button_back))) else null
        val messageDialogModel = MessageDialogModel(
                message = message,
                description = null,
                optionsButtonText = optionButtons,
                cancelButtonText = null,
                code = MessageDialogCodes.appCannotRunRuntime
        )
        dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        (dialogFragment as MessageDialogFragment).mListener = object :
                MessageDialogFragmentListener {
            override fun onResult(answer: MessageDialogAction): Boolean {
                limitationDialogShown = false
                if (answer is MessageDialogAction.Result && answer.type == OPTION_A) {
                    (context as? MainActivityAction)?.resolveDialogBackOptionPress()
                }
                if (answer is MessageDialogAction.Event && isInPlayerMode) {
                    return when (answer.keyEvent.keyCode) {
                        KeyEvent.KEYCODE_CHANNEL_UP -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToNextChannel.invoke()
                            true
                        }
                        KeyEvent.KEYCODE_CHANNEL_DOWN -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToPreviousChannel.invoke()
                            true
                        }
                        KeyEvent.KEYCODE_BACK -> {
                            dialogFragment?.dismiss()
                            dialogFragment = null
                            goToLastPlayedChannel.invoke()
                            true
                        }
                        in KeyEvent.KEYCODE_0..KeyEvent.KEYCODE_9 -> {
                            dialogFragment?.activity?.dispatchKeyEvent(answer.keyEvent)
                            true
                        }
                        else -> return false
                    }
                }
                return false
            }
        }
        (dialogFragment as MessageDialogFragment).mDismissListener = object : MessageDialogDismissListener {
            override fun onDismiss() {
                limitationDialogShown = false
            }
        }
        dialogFragment?.let {
            NavigatorCommon.getInstance().navigateDialog(it)
        }
    }
}