package com.twentyfouri.tvlauncher.common.utils

import android.content.Context
import com.twentyfouri.smartmodel.model.error.*
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import java.io.InterruptedIOException
import java.net.ConnectException
import java.net.SocketTimeoutException
import java.nio.channels.InterruptedByTimeoutException
import javax.net.ssl.HttpsURLConnection

object ErrorStringMapper {

    fun getErrorString(e: Exception, context: Context): String {
        val resourceRepository = ResourceRepository(context)
        return when {
            e is NotFoundException ->
                e.code
            e is AppCannotRunException ->
                resourceRepository.getString(R.string.app_cannot_run)
            e is WrongInputException || e is InvalidCredentialsException ->
                resourceRepository.getString(R.string.authentication_failure_dialog_message)
            e is RecordingException -> resourceRepository.getString(R.string.epg_recordings_unavailable_with_code, e.code)
            e is GeneralApiException ->
                translateException(e.code, resourceRepository)
            e is InternetConnectionException -> {
                if (e.cause is InterruptedByTimeoutException || e.cause is InterruptedIOException)
                    resourceRepository.getString(R.string.connection_time_out_issues_toast_message)
                else resourceRepository.getString(R.string.connectivity_issues_toast_message)
            }
            e is SocketTimeoutException || e.cause is SocketTimeoutException ->
                resourceRepository.getString(R.string.connection_time_out_issues_toast_message)
            e is ConnectException ->
                processConnectExceptionMessage(e, context)
            e is java.lang.IllegalStateException || e is IllegalArgumentException ->
                resourceRepository.getString(R.string.api_connection_issue)
            e is javax.net.ssl.SSLException -> resourceRepository.getString(R.string.ssl_exception)
            else -> resourceRepository.getString(R.string.unhandled_exception_toast_message)
        }
    }

    private fun processConnectExceptionMessage(e: ConnectException, context: Context):String {
        val resourceRepository = ResourceRepository(context)
        val message = e.message ?: return resourceRepository.getString(R.string.connectivity_issues_toast_message)
        return when {
            message.toLowerCase().contains("timed out") -> resourceRepository.getString(R.string.connection_time_out_issues_toast_message)
            else -> resourceRepository.getString(R.string.connectivity_issues_toast_message)
        }
    }

    private fun translateException(code: String, resourceRepository: ResourceRepository): String{
       return when (code){
            HttpsURLConnection.HTTP_BAD_REQUEST.toString() -> resourceRepository.getString(R.string.http_bad_request, code)
            HttpsURLConnection.HTTP_FORBIDDEN.toString() -> resourceRepository.getString(R.string.http_forbidden, code)
            HttpsURLConnection.HTTP_NOT_FOUND.toString() -> resourceRepository.getString(R.string.http_not_found, code)
            HttpsURLConnection.HTTP_CLIENT_TIMEOUT.toString() -> resourceRepository.getString(R.string.http_timeout, code)
            HttpsURLConnection.HTTP_INTERNAL_ERROR.toString() -> resourceRepository.getString(R.string.http_internal_server_error, code)
            HttpsURLConnection.HTTP_UNAVAILABLE.toString(),
            HttpsURLConnection.HTTP_GATEWAY_TIMEOUT.toString() ->resourceRepository.getString(R.string.http_server_unavailable, code)
            else -> resourceRepository.getString(R.string.http_error_other, code)
        }
    }
}