package com.twentyfouri.tvlauncher.common.utils

import android.annotation.SuppressLint
import android.app.ActivityManager
import android.content.Context
import android.content.IntentFilter
import android.os.Build
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.receiver.SerialNumberReceiver
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import timber.log.Timber
import java.io.RandomAccessFile
import java.text.DecimalFormat
import java.text.DecimalFormatSymbols
import java.util.Locale
import java.util.Timer
import kotlin.concurrent.fixedRateTimer
import kotlin.math.ln
import kotlin.math.pow

interface InfoUpdated {
    suspend fun onInfoUpdated()
}

data class DeviceInfoConfig(val launcherPackage: String, val wizardPackage: String)

class DeviceInfo(context: Context, val infoUpdated: InfoUpdated, val deviceInfoConfig: DeviceInfoConfig) : SerialNumberReceiverHelper {

    private val CPU_INFO_DIR = "/sys/devices/system/cpu/"
    private var repeatTimer: Timer? = null

    @SuppressLint("MissingPermission")
    var serialNo: String = getSerialNo(context)

    val deviceId = DeviceId.getDeviceId(context)
    var serialNumberReceiver: SerialNumberReceiver? = null
    val memoryInfo = ActivityManager.MemoryInfo()
    var memTotalFormatted = ""
    var memAvailableFormatted = ""
    var memLow = false

    var memTotal: Long = 0L
    var memAvailable: Long = 0L

    var launcherVersion = ""
    var wizardVersion = ""

    var numCores = 0
    var maxFreq = getMinMaxFreq(0).second
    var averageFreq = 0L

    val osVersion = deviceOsVersion
    val firmware = Flavor().firmwareVersion

    init {
        val packageInfoLauncher = context.packageManager.getPackageInfo(deviceInfoConfig.launcherPackage, 0)
        launcherVersion = "${packageInfoLauncher.versionName} (${packageInfoLauncher.versionCode})"
        val packageInfoWizard = context.packageManager.getPackageInfo(deviceInfoConfig.wizardPackage, 0)
        wizardVersion = "${packageInfoWizard.versionName} (${packageInfoWizard.versionCode})"
        numCores = getNumberOfCores()
        CoroutineScope(Dispatchers.IO).launch { update(context) }
    }

    suspend fun update(context: Context) {
        val activityManager = context.getSystemService(Context.ACTIVITY_SERVICE) as? ActivityManager
        activityManager?.getMemoryInfo(memoryInfo)
        memTotal = memoryInfo.totalMem
        memAvailable = memoryInfo.availMem
        memTotalFormatted = convertBytesToMega(memoryInfo.totalMem)
        val memAvailablePercent = (memoryInfo.availMem.toDouble() / memoryInfo.totalMem.toDouble() * 100.0).toInt()
        memAvailableFormatted = "${convertBytesToMega(memoryInfo.availMem)} ($memAvailablePercent%)"
        memLow = memoryInfo.lowMemory

        var freqCount = 0L
        for(i in 0 until numCores) {
            freqCount += getCurrentFreq(i)
        }
        averageFreq = freqCount/numCores

        infoUpdated.onInfoUpdated()
    }

    fun startAutoUpdate(context: Context, period: Long = 1000L) {
        repeatTimer?.cancel()
        repeatTimer = null
        repeatTimer = fixedRateTimer("DeviceInfoTimer", true, 0, period) {
            CoroutineScope(Dispatchers.IO).launch {
                update(context)
            }
        }
    }

    fun stopAutoUpdate() {
        repeatTimer?.cancel()
        repeatTimer = null
    }

    suspend fun toString(context: Context): String {
        val result = StringBuilder()
        result.appendln("${context.getString(R.string.di_serial_number)} $serialNo")
        result.appendln("${context.getString(R.string.di_device_id)} $deviceId")
        result.appendln()
        result.appendln("${context.getString(R.string.di_mem_total)} $memTotalFormatted")
        result.appendln("${context.getString(R.string.di_mem_avail)} $memAvailableFormatted")
        result.appendln("${context.getString(R.string.di_mem_low)} $memLow")
        result.appendln()
        result.appendln("${context.getString(R.string.di_cpu_cores)} $numCores")
        result.appendln("${context.getString(R.string.di_cpu_freq)} $maxFreq Hz")
        result.appendln("${context.getString(R.string.di_cpu_average)} $averageFreq Hz")
        result.appendln()
        result.appendln("${context.getString(R.string.di_launcher)} $launcherVersion")
        result.appendln("${context.getString(R.string.di_wizard)} $wizardVersion")
        result.appendln()
        result.appendln("${context.getString(R.string.di_os_version)} ${osVersion.version} - ${osVersion.codename} (API ${osVersion.api})")
        result.appendln("${context.getString(R.string.di_firmware)} ${firmware.await()}")

        return result.toString()
    }

    @SuppressLint("MissingPermission")
    private fun getSerialNo(context: Context): String {
        return if (Flavor().getSerialNumberFromBroadcast()) {
            val serialNumberIntentFilter = IntentFilter()
            serialNumberIntentFilter.addAction(SerialNumberReceiver.INTENT_FILTER_ACTION)
            serialNumberReceiver = SerialNumberReceiver(this, false)
            context.registerReceiver(serialNumberReceiver, serialNumberIntentFilter)
            context.getString(R.string.di_conn_pending)
        } else {
            try {
                Build.getSerial()
            } catch (e: SecurityException) {
                Timber.d("android.permission.READ_PHONE_STATE not granted, serialNo not obtained")
                context.getString(R.string.di_phone_permission)
            }
        }
    }

    override fun onSerialNumberReceivedWithResult(result: String?, context: Context?) {
        try {
            context?.unregisterReceiver(serialNumberReceiver)
        } catch (e: IllegalArgumentException) {
            Timber.d(e.stackTraceToString())
        }
        serialNumberReceiver = null
        serialNo = result ?: "null"
    }

    fun humanReadableByteCount(bytes: Long): String {
        val unit = 1024
        if (bytes < unit) return "$bytes B"
        val exp = (ln(bytes.toDouble()) / ln(unit.toDouble())).toInt()
        val pre = "KMGTPE"[exp - 1]
        return String.format(Locale.US, "%.2f %sB", bytes / unit.toDouble().pow(exp.toDouble()), pre)
    }

    fun convertBytesToMega(bytes: Long): String {
        val megaBytes = bytes.toDouble() / (1024.0 * 1024.0)
        val df = DecimalFormat("#.##", DecimalFormatSymbols(Locale.US))

        return "${df.format(megaBytes)} MB"
    }

    fun getNumberOfCores(): Int {
        return Runtime.getRuntime().availableProcessors()
    }

    fun getCurrentFreq(coreNumber: Int): Long {
        val currentFreqPath = "${CPU_INFO_DIR}cpu$coreNumber/cpufreq/scaling_cur_freq"
        return try {
            RandomAccessFile(currentFreqPath, "r").use { it.readLine().toLong() / 1000 }
        } catch (e: Exception) {
            Timber.tag("SYSDIA").d("get freq $coreNumber: ${e.message}")
            -1
        }
    }

    fun getMinMaxFreq(coreNumber: Int): Pair<Long, Long> {
        val minPath = "${CPU_INFO_DIR}cpu$coreNumber/cpufreq/cpuinfo_min_freq"
        val maxPath = "${CPU_INFO_DIR}cpu$coreNumber/cpufreq/cpuinfo_max_freq"
        return try {
            val minMhz = RandomAccessFile(minPath, "r").use { it.readLine().toLong() / 1000 }
            val maxMhz = RandomAccessFile(maxPath, "r").use { it.readLine().toLong() / 1000 }
            Pair(minMhz, maxMhz)
        } catch (e: Exception) {
            Timber.tag("SYSDIA").d("get min/max $coreNumber: ${e.message}")
            Pair(-1, -1)
        }
    }

}

