package com.twentyfouri.tvlauncher.common.ui.pindialog

import android.view.KeyEvent
import android.view.View
import android.widget.Toast
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.ViewModel
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.error.SmartApiException
import com.twentyfouri.smartmodel.model.error.WrongInputException
import com.twentyfouri.smartmodel.model.error.WrongPinException
import com.twentyfouri.tvlauncher.common.Flavor
import com.twentyfouri.tvlauncher.common.R
import com.twentyfouri.tvlauncher.common.ui.TvLauncherToast
import com.twentyfouri.tvlauncher.common.widgets.PinEntryView
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.collect
import org.koin.core.component.KoinComponent
import org.koin.core.component.get

class PinDialogViewModel : ViewModel(), KoinComponent {

    var model: PinDialogModel? = null
        set(value) {
            field = value
            initView()
        }

    var view: PinDialogView? = null

    private fun initView() {
        model?.let {
            description.value = it.messageDescription
            message.value = it.message
        }
    }

    // region Binded

    val message = MutableLiveData<String>()
    val description = MutableLiveData<String>()

    fun getOnPinEnteredListener(): PinEntryView.OnPinEnteredListener {
        return object : PinEntryView.OnPinEnteredListener {
            override fun onPinEntered(v: View, str: CharSequence) {
                CoroutineScope(Dispatchers.Main).launch {
                    try {
                        get<FlowSmartApi>().verifyParentalPin(str.toString()).collect()
                        view?.onResult(true, null)
                    } catch (e: SmartApiException) {
                        super.onPinEntered(v, str)
                        resolveOnPinEnteredException(v,e)
                    }
                }
            }

            override fun onChannelChangedBehind(keyCode: Int) {
                view?.onResult(false, keyCode)
            }
        }
    }

    fun getOnPurchasePinEnteredListener(): PinEntryView.OnPinEnteredListener {
        return object : PinEntryView.OnPinEnteredListener {
            override fun onPinEntered(v: View, str: CharSequence) {
                CoroutineScope(Dispatchers.IO).launch {
                    try {
                        Flavor().verifyPurchasePin(
                            get(), str.toString()
                        )
                        withContext(Dispatchers.Main){
                            view?.onResult(true, null)
                        }
                    } catch (e: GeneralApiException) {
                        super.onPinEntered(v, str)
                        resolveOnPinEnteredException(v,e)
                    }
                }
            }

            override fun onChannelChangedBehind(keyCode: Int) {
                view?.onResult(false, keyCode)
            }
        }
    }

    private fun resolveOnPinEnteredException(v: View, e: SmartApiException){
        when (e) {
            is WrongInputException, is WrongPinException -> {
                GlobalScope.launch(Dispatchers.Main) {
                    TvLauncherToast.makeText(
                        v.context,
                        v.context.getString(R.string.error_wrong_pin),
                        Toast.LENGTH_LONG
                    )?.show()
                }
            }
            else -> {
                GlobalScope.launch(Dispatchers.Main) {
                    val code = if (e is GeneralApiException){
                        "${e.code}, "
                    } else ""
                    TvLauncherToast.makeText(
                        v.context,
                        "${v.context.getString(R.string.backend_error)}: $code${e.message}",
                        Toast.LENGTH_LONG
                    )?.show()
                }
                view?.onResult(
                    false,
                    KeyEvent.KEYCODE_BACK
                ) //Different error should go to previous channel
            }

        }
    }
    // endregion
}