package com.twentyfouri.tvlauncher.common.ui.fading

import android.content.Context
import android.util.AttributeSet
import android.view.View
import androidx.appcompat.widget.AppCompatTextView
import com.twentyfouri.tvlauncher.common.R

/*
 * Animation in this class is based on text change. When setText is called then animation will start.
 * First setText update text immediately because it is needed during first layout/measure.
 */
class FadingTextView @JvmOverloads constructor(
        context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : AppCompatTextView(context, attrs, defStyleAttr), FadingViewInterface {

    private var newText: CharSequence? = null
    private var newBufferType: BufferType? = null
    private var setOnce = false
    private var fadingController: FadingController? = null
    private var fadingControllerId: Int = -1

    init {
        val a = context.theme.obtainStyledAttributes(attrs, R.styleable.FadingView, 0, 0)
        try {
            fadingControllerId = a.getResourceId(R.styleable.FadingView_fadingController, -1)
        } finally {
            a.recycle()
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        //in case controller id is not set we expect this view should not be part of some group so create its own controller
        fadingController = if(fadingControllerId == -1) {
            FadingController()
        } else {
            (rootView.findViewById<View>(fadingControllerId) as? FadingControllerView)?.fadingController
        }
        fadingController?.addView(this)
    }

    override fun onDetachedFromWindow() {
        if(fadingControllerId == -1) {
            fadingController?.detach()
        } else {
            fadingController?.removeView(this)
        }
        super.onDetachedFromWindow()
    }

    override fun setText(text: CharSequence?, type: BufferType?) {
        if(setOnce) {
            newText = text
            newBufferType = type
            fadingController?.update()
        } else {
            super.setText(text, type)
            setOnce = true
        }
    }

    override fun getText(): CharSequence? {
        return if(setOnce) {
            newText
        } else {
            super.getText()
        }
    }

    override fun onFadedOut() {
        super.setText(newText, newBufferType)
    }

    override fun getControllerId(): Int {
        return fadingControllerId
    }
}