package com.twentyfouri.tvlauncher.common.ui.fading

import android.content.Context
import android.util.AttributeSet
import android.view.View
import androidx.appcompat.widget.AppCompatImageView
import com.twentyfouri.tvlauncher.common.R

/*
 * Animation in this class is based on visibility. So when visibility change then animation is started.
 * This class will not animate when image/source is changed.
 */
class FadingImageView @JvmOverloads constructor(
        context: Context, attrs: AttributeSet? = null, defStyleAttr: Int = 0
) : AppCompatImageView(context, attrs, defStyleAttr), FadingViewInterface {

    private var fadingController: FadingController? = null
    private var fadingControllerId: Int = -1
    var newVisibility: Int = View.VISIBLE
    var setOnce = false

    init {
        val a = context.theme.obtainStyledAttributes(attrs, R.styleable.FadingView, 0, 0)
        try {
            fadingControllerId = a.getResourceId(R.styleable.FadingView_fadingController, -1)
        } finally {
            a.recycle()
        }
    }

    override fun onAttachedToWindow() {
        super.onAttachedToWindow()
        //in case controller id is not set we expect this view should not be part of some group so create its own controller
        fadingController = if(fadingControllerId == -1) {
            FadingController()
        } else {
            (rootView.findViewById<View>(fadingControllerId) as? FadingControllerView)?.fadingController
        }
        fadingController?.addView(this)
    }

    override fun onDetachedFromWindow() {
        if(fadingControllerId == -1) {
            fadingController?.detach()
        } else {
            fadingController?.removeView(this)
        }
        super.onDetachedFromWindow()
    }

    override fun setVisibility(visibility: Int) {
        if (setOnce) {
            if (newVisibility == visibility) return
            newVisibility = visibility
            fadingController?.update()
        } else {
            super.setVisibility(visibility)
            setOnce = true
        }
    }

    override fun onFadedOut() {
        super.setVisibility(newVisibility)
    }

    override fun getControllerId(): Int {
        return fadingControllerId
    }

}