package com.twentyfouri.tvlauncher.common

import android.content.ContentValues
import android.content.Context
import android.net.Uri
import android.util.Log
import com.google.gson.Gson
import com.twentyfouri.androidcore.multilanguage.language.LanguageInfoAuto
import com.twentyfouri.androidcore.multilanguage.localization.LocalizationNative
import com.twentyfouri.androidcore.multilanguage.localization.LocalizationUninitialized
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.backstage.BackstageConfiguration
import com.twentyfouri.smartmodel.backstage.BackstageFlowSmartApi
import com.twentyfouri.smartmodel.backstage.mapper.BackstageFaqDetectorDefault
import com.twentyfouri.smartmodel.backstage.reference.BackstageUserProfileReference
import com.twentyfouri.smartmodel.caching.LowMemoryHandler
import com.twentyfouri.smartmodel.configuration.Ads
import com.twentyfouri.smartmodel.model.error.GeneralApiException
import com.twentyfouri.smartmodel.model.user.SmartSavedSession
import com.twentyfouri.smartmodel.model.user.SmartSessionPersistence
import com.twentyfouri.smartmodel.model.user.SmartUserProfileReference
import com.twentyfouri.smartmodel.model.watchlist.SmartContinueWatchingNull
import com.twentyfouri.smartmodel.model.watchlist.SmartWatchlistNull
import com.twentyfouri.smartmodel.serialization.SmartDataObject
import com.twentyfouri.smartmodel.util.AdTagProvider
import com.twentyfouri.smartmodel.util.SmartImageSizing
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.extensions.ifNotNull
import com.twentyfouri.tvlauncher.common.utils.DeviceId
import kotlinx.coroutines.runBlocking

class Flavor : FlavorBase() {

    override fun getSmartApi(applicationContext: Context?, setupDataRepo: SetupDataRepository): FlowSmartApi {
        applicationContext ?: throw IllegalStateException("Context null when creating SmartApi")
        val udid = DeviceId.getDeviceId(applicationContext)
        return buildBackstageSmartApi(applicationContext, setupDataRepo, udid)
    }

    private fun buildBackstageSmartApi(
            applicationContext: Context,
            setupDataRepo: SetupDataRepository,
            udid: String
    ) = BackstageFlowSmartApi( BackstageKtSmartApi (
            configuration = BackstageConfiguration().apply {
                endpoint = ENDPOINT
                deviceType = DEVICE_TYPE
                serviceId = SERVICE_ID
                applicationId = APPLICATION_ID
            },
            adTagProvider = AdTagProvider(Ads(), udid),
            localHistory = SmartContinueWatchingNull(),
            localFavorites = SmartWatchlistNull(),
            localization = LocalizationNative(
                    applicationContext,
                    LanguageInfoAuto("EN")
            ),
            imagesTransformer = SmartImageSizing.DisabledTransformer(),
            lowMemoryHandler = LowMemoryHandler(applicationContext),
            deviceId = udid,
            sessionPersistence = object : SmartSessionPersistence {
                override fun loadSession(): SmartSavedSession? = runBlocking {
                    Timber.tag("SessionStorage").d("SmartSessionPersistence loadSession")
                    val setupData = setupDataRepo.getSetupDataObject()
                    Timber.tag("SessionStorage").d("\"SmartSessionPersistence loadSession json: ${setupData?.sessionData}\"")
                    val sessionFromProvider = getSavedSession(setupData?.sessionData)
                    sessionFromProvider.ifNotNull { return@runBlocking sessionFromProvider }

                    return@runBlocking null
                }

                override fun saveSession(session: SmartSavedSession?) {
                    val json: String? = session?.let { Gson().toJson(it) }
                    val values = ContentValues()
                    val uri: Uri = Uri.parse(applicationContext.getString(R.string.credentials_uri))
                    values.put("sessionData", json)
                    applicationContext.contentResolver.insert(uri, values)
                }
            },
            faqDetector = BackstageFaqDetectorDefault()
    ) )

    override suspend fun performPing(smartApi: FlowSmartApi) { }

    override fun getSavedSession(setupDataString: String?): SmartSavedSession? = null

    override fun getUserProfileReference(userName: String): SmartUserProfileReference = BackstageUserProfileReference(userName, null)

    override fun gdprProfileChangeObject(isGdrpAccepted: Boolean?): SmartDataObject = SmartDataObject()

    override fun isGdprAcceptedProfile(extras: SmartDataObject): Boolean? = null

    override fun getUsername(sessionData: String?): String? = null

    override fun getErrorMessage(e: GeneralApiException, applicationContext: Context): String = ""

    override suspend fun verifyPurchasePin(smartApi: FlowSmartApi, purchasePin: String) { }

    override fun clearSessionData(context: Context) { }

    companion object {
        private const val DEFAULT_DEVICE_ID = "defaultDeviceId"
        private const val ENDPOINT = "https://backstage-api.com/"
        private const val DEVICE_TYPE = "Tv"
        private const val SERVICE_ID = "5e0ad1b0-515e-11e9-a7ed-371ac744bd33"
        private const val APPLICATION_ID = "5dc484f0-d7ec-11ea-9758-096b005f3efd"
    }
}