package com.twentyfouri.tvlauncher

import android.content.Context
import android.net.Uri
import android.view.KeyEvent
import android.view.KeyEvent.*
import com.google.android.exoplayer2.drm.MediaDrmCallback
import com.google.gson.Gson
import com.google.gson.JsonElement
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.media.SmartPlayerEvent
import com.twentyfouri.smartmodel.model.media.SmartRestrictionType
import com.twentyfouri.smartmodel.model.menu.SmartMenu
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartPurchaseState
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.smartmodel.model.payment.SmartSubscriptionReference
import com.twentyfouri.smartmodel.model.user.SmartSessionState
import com.twentyfouri.smartmodel.persistence.SSTResult
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.smartmodel.smartvideo.SmartVideoFlowSmartApi
import com.twentyfouri.smartmodel.smartvideo.SmartVideoReferenceFactory
import com.twentyfouri.smartmodel.smartvideo.mapper.SmartVideoMediaDetailMapper
import com.twentyfouri.smartmodel.smartvideo.mapper.SmartVideoMediaItemMapper
import com.twentyfouri.smartmodel.smartvideo.mapper.SmartVideoMenuMapper
import com.twentyfouri.smartmodel.smartvideo.reference.*
import com.twentyfouri.smartmodel.util.last
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.HardcodedMenuMapper
import com.twentyfouri.tvlauncher.data.HardcodedPageMapper
import com.twentyfouri.tvlauncher.data.HardcodedPageReference
import com.twentyfouri.tvlauncher.ui.EpgFragment
import com.twentyfouri.tvlauncher.utils.KeyEventAction
import com.twentyfouri.tvlauncher.viewmodels.FilterViewModel
import com.twentyfouri.tvlauncher.viewmodels.KijkwijzerIconsViewModel
import com.twentyfouri.tvlauncher.viewmodels.MediaRestrictionIconsViewModel
import com.vualto.vudrm.HttpKidSource
import com.vualto.vudrm.kidplugin.BuildConfig
import com.vualto.vudrm.widevine.AssetConfiguration
import com.vualto.vudrm.widevine.WidevineCallback
import com.vualto.vudrm.widevine.vudrm
import kotlinx.coroutines.flow.Flow
import org.joda.time.DateTime
import org.joda.time.DateTimeConstants.MILLIS_PER_DAY
import org.joda.time.Seconds
import java.net.URL
import java.util.*

class Flavor : FlavorBase() {

    companion object {
        private const val GenreToFilterAdultChannels = "adult"
        private const val TIMEZONE_ID = "Europe/Amsterdam"
    }

    override val bookmarkResetThresholdMillis: Int = 180000 // 3 minutes in ms

    override fun getBookmarkReference(detail: SmartMediaDetail): SmartMediaReference? {
        val pReference = (detail.programReference as? SmartVideoMediaReference) ?: return null
        return when (detail.type) {
            SmartMediaType.RECORDING -> SmartVideoMediaReference(
                id = pReference.id,
                type = SmartMediaType.RECORDING
            )
            SmartMediaType.LIVE_EVENT -> SmartVideoMediaReference(
                id = pReference.id,
                type = SmartMediaType.LIVE_EVENT
            )
            else -> null
        }
    }

    override val useBookmarks: Boolean = true

    override val isRecordingAllowed: Boolean = true

    override val shouldHomeButtonFocusHomeTab = true

    override fun shouldHandleGuideButton(): Boolean = true

    override val shouldPlayerHandleLiveButton: Boolean = true

    override fun isTunnelingEnabled(): Boolean = false

    override val getEpgConfigValues: EpgFragment.EPGConfigValues = EPGConfigValuesYoufone()

    override val shouldStartStreamTimeout: Boolean
        get() = true

    override val isLiveRecordingAllowed: Boolean
        get() = true

    override fun compareRecordingWithEvent(
        recording: SmartMediaItem?,
        event: SmartMediaItem?
    ): Boolean {
        event?.programReference ?: return false
        return recording?.programReference == event.programReference
    }

    //TODO this is temporary solution - will be removed once we start using EPG database caching
    override val defaultWindowStartTime: DateTime = TimeProvider.now().minusHours(4)
    override val defaultControlsWindows = EpgRepository.PLAYER_CONTROLS_EPG_WINDOW_EXTENDED
    override fun getEpgDataFilteredSorted(epgData: MutableList<SmartMediaItem>, channelReference: SmartMediaReference): List<SmartMediaItem> {
        epgData
            .sortBy { it.startDate }
        return epgData
            .filter { it.endDate!! > TimeProvider.now()}
    }

    override fun allowDisplayOfExternalRows() = BuildConfig.DEBUG

    override suspend fun isLoggedIn(smartApi: FlowSmartApi): Boolean {
        return when (smartApi.restoreSessionImmediate().last()) {
            SmartSessionState.LOGGED_IN -> true
            else -> false
        }
    }

    override fun getPageReference(type: PageType, menus: List<SmartMenu>?) = when (type) {
        PageType.MAIN -> SmartVideoPageReference(SmartVideoPageReference.Type.MAIN)
        PageType.EPG -> SmartVideoPageReference(SmartVideoPageReference.Type.EPG)
        PageType.GRID -> SmartVideoPageReference(SmartVideoPageReference.Type.GRID)
        PageType.APPS -> HardcodedPageReference(HardcodedPageReference.Type.APPS)
    }

    override fun getPlaylistReference(type: PlaylistType) = getPlaylistReference(type, null)

    override fun getPlaylistReference(
        type: PlaylistType,
        externalReference: SmartMediaReference?
    ): SmartPlaylistReference {
        return when (type) {
            PlaylistType.ALL_CHANNELS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.ALL_CHANNELS)
            PlaylistType.ON_NOW -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.ON_NOW_PROGRAMS)
            PlaylistType.DO_NOT_MISS_CHANNELS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.DO_NOT_MISS_PROGRAMS)
            PlaylistType.RECOMMENDED_APPS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.RECOMMENDED_APPS)
            PlaylistType.UPCOMING_PROGRAMS -> SmartVideoPlaylistReference(
                SmartVideoPlaylistReference.Type.UPCOMING_PROGRAMS, listOfNotNull(externalReference))
            PlaylistType.CATCHUP_PROGRAMS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.CATCHUP_PROGRAMS)
            PlaylistType.RECENT_RECORDINGS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.RECORDINGS,null, externalReference)
            PlaylistType.RECORDINGS -> SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.RECORDINGS)
            else -> throw IllegalArgumentException("Incorrect playlist type")
        }
    }

    override fun getBookmarkProgress(positionInSeconds: Int, startDate: DateTime?, endDate: DateTime?, durationInSeconds: Int?): Int {
        val adjustedPositionInSeconds = positionInSeconds - adsDurationBeforeRecordingMillis / 1000
        val secondsBetween = if (startDate != null && endDate != null) Seconds.secondsBetween(startDate, endDate).seconds
        else durationInSeconds ?: return 0
        return ((if (adjustedPositionInSeconds > 0) adjustedPositionInSeconds else 0) * 100) / secondsBetween
    }

    override fun getGenreFilterReference(title: String, type: PlaylistType?) = SmartVideoPlaylistFilterReferenceGenre("TODO-genreId", title)

    override fun convertMediaReferenceToDetail(reference: SmartMediaReference) =
        if (reference is SmartVideoMediaReference) SmartVideoMediaDetailMapper.convertSmartVideoMediaReference(reference)
        else SmartMediaDetail(reference, SmartMediaType.UNKNOWN)

//    override fun convertDetailToRecording(mediaDetail: SmartMediaDetail, autoRecording: Boolean) = when (autoRecording) {
//        false -> SmartMediaItem(mediaDetail.reference, mediaDetail.type).apply {
//            channelReference = mediaDetail.channelReference
//            recordingReference = mediaDetail.recordingReference
//            startDate = mediaDetail.startDate
//            programReference = mediaDetail.programReference
//        }
//        true -> SmartMediaItem(mediaDetail.seriesReference!!, SmartMediaType.SERIES).apply {
//            channelReference = mediaDetail.channelReference
//            recordingReference = mediaDetail.recordingReference
//        }
//    }
    override fun convertDetailToRecording(mediaDetail: SmartMediaDetail, autoRecording: Boolean): SmartMediaItem {
        when {
            autoRecording -> {
                return SmartMediaItem(
                    mediaDetail.recordingReference
                        ?: SmartVideoMediaReference("unknown", SmartMediaType.AUTO_RECORDING),
                    SmartMediaType.AUTO_RECORDING
                ).apply {
                    channelReference = mediaDetail.channelReference
                    seriesReference = mediaDetail.seriesReference
                }
            }
            mediaDetail.type == SmartMediaType.RECORDING -> {
                return mediaDetail
            }
            else -> {
                return SmartMediaItem(
                    mediaDetail.recordingReference
                        ?: SmartVideoMediaReference("unknown", SmartMediaType.RECORDING),
                    SmartMediaType.RECORDING
                ).apply {
                    channelReference = mediaDetail.channelReference
                    programReference = mediaDetail.programReference
                    liveEventReference = mediaDetail.reference
                    startDate = mediaDetail.startDate
                }
            }
        }
    }

    override fun getChannelId(mediaItem: SmartMediaItem) = (mediaItem.reference as SmartVideoMediaReference).id

    override fun getImageOfType(mediaItem: SmartMediaItem, imageType: ImageType): SmartImages {
        return when(imageType) {
            ImageType.DARK -> {
                if (mediaItem.type == SmartMediaType.LIVE_CHANNEL) {
                    mediaItem.overlayImages?.get(SmartVideoMediaItemMapper.KEY_CHANNEL_ICON_DARK)
                        ?: mediaItem.overlayImages?.get(SmartVideoMediaItemMapper.KEY_CHANNEL_ICON)
                        ?: SmartImages()
                } else {
                    mediaItem.images
                }
            }
            ImageType.LIGHT -> {
                if (mediaItem.type == SmartMediaType.LIVE_CHANNEL) {
                    mediaItem.overlayImages?.get(SmartVideoMediaItemMapper.KEY_CHANNEL_ICON_LIGHT)
                        ?: mediaItem.overlayImages?.get(SmartVideoMediaItemMapper.KEY_CHANNEL_ICON)
                        ?: SmartImages()
                } else {
                    mediaItem.images
                }
            }
            ImageType.OVERLAY -> {
                if (mediaItem.type == SmartMediaType.LIVE_CHANNEL)
                    mediaItem.images
                else
                    mediaItem.overlayImages?.get(SmartVideoMediaItemMapper.KEY_CHANNEL_ICON) ?: SmartImages()
            }
        }
    }

    override fun getMenuType(menuType: String) = when (menuType) {
        HardcodedMenuMapper.Type.SEARCH.name -> MenuType.SEARCH
        HardcodedMenuMapper.Type.NOTIFICATIONS.name -> MenuType.NOTIFICATIONS
        HardcodedMenuMapper.Type.VOICE.name -> MenuType.VOICE
        HardcodedMenuMapper.Type.SETTINGS.name -> MenuType.SETTINGS
        HardcodedMenuMapper.Type.APPS.name -> MenuType.APPS
        HardcodedMenuMapper.Type.PERSONAL.name -> MenuType.PERSONAL
        SmartVideoMenuMapper.Type.HOME.name -> MenuType.HOME
        SmartVideoMenuMapper.Type.EPG.name -> MenuType.EPG
        SmartVideoMenuMapper.Type.GRID.name -> MenuType.GRID
        else -> MenuType.UNKNOWN
    }

    override fun getPlaylistTypeForLocalization(sectionStyle: String): PlaylistType {
        when (val sectionStyleFromBase = super.getPlaylistTypeForLocalization(sectionStyle)) {
            PlaylistType.UNKNOWN -> { //super implementation was NOT able to identify the section style
                return when (sectionStyle) {
                    SmartVideoPlaylistReference.Type.DO_NOT_MISS_PROGRAMS.name -> PlaylistType.DO_NOT_MISS_CHANNELS
                    SmartVideoPlaylistReference.Type.ON_NOW_PROGRAMS.name -> PlaylistType.ON_NOW
                    SmartVideoPlaylistReference.Type.ALL_CHANNELS.name -> PlaylistType.ALL_CHANNELS
                    SmartVideoPlaylistReference.Type.RECOMMENDED_APPS.name -> PlaylistType.RECOMMENDED_APPS
                    SmartVideoPlaylistReference.Type.UPCOMING_PROGRAMS.name -> PlaylistType.UPCOMING_PROGRAMS
                    SmartVideoPlaylistReference.Type.CATCHUP_PROGRAMS.name -> PlaylistType.CATCHUP_PROGRAMS
                    SmartVideoPlaylistReference.Type.RECORDINGS.name, PlaylistType.RECENT_RECORDINGS.name -> PlaylistType.RECENT_RECORDINGS
                    SmartVideoPlaylistReference.Type.LAST_WATCHED.name -> PlaylistType.LAST_WATCHED
                    SmartVideoPlaylistReference.Type.GRID.name -> PlaylistType.GRID
                    else -> PlaylistType.UNKNOWN
                }
            }
            else -> { //super implementation WAS able to identify the section style
                return sectionStyleFromBase
            }
        }
    }

    override fun getPageSize(section: SmartPageSection) = when(getPlaylistType(section)) {
        PlaylistType.LAST_WATCHED -> 20 //value for Youfone
        PlaylistType.GRID -> 5*5 //value for Youfone GRID, currently 25 items per page
        else -> defaultPageSize
    }

    override fun getPlaylistTotalSizeMax(section: SmartPageSection) = when(getPlaylistType(section)) {
        PlaylistType.GRID -> 4 * getPageSize(section) //value for Youfone GRID, currently 4 pages
        else -> Int.MAX_VALUE
    }

    override fun hasAnotherPage(section: SmartPageSection, previousPageTotalCount: Int): Boolean {
        return when (getPlaylistType(section)){
            PlaylistType.GRID -> {
                (section.selectedOptions?.totalCount != 0  // first page already returned 0 items.
                        // previous page returned less then pagingOffset number of items.
                        && section.selectedOptions?.pagingOffset?.mod(getPageSize(section)) == 0
                        // previous page returned 0 items.
                        && previousPageTotalCount != section.selectedOptions?.totalCount
                        // limit of loaded items reached.
                        && (section.selectedOptions?.totalCount ?: 0) < getPlaylistTotalSizeMax(section)  )
            }
            else -> {
                super.hasAnotherPage(section, previousPageTotalCount)
            }
        }
    }

    override fun isCatchupStreamAvailable(item: SmartMediaItem): Boolean {
        //val stream = (reference as? SmartVideoMediaReference)?.editions?.get(0)?.streams?.get(0) as? SmartVideoStreamReference
        //return stream?.streamId?.equals("null")?.not() ?: false
        val endDate = item.endDate
//        Log.d("FlowApi", "isCatchupStreamAvailable $endDate")
//        Log.d("FlowApi", "isCatchupStreamAvailable ${TimeProvider.now().minusSeconds(item.catchupOffsetInSeconds)}")
//        Log.d("FlowApi", "isCatchupStreamAvailable ${item.restrictions.find { it.type == SmartRestrictionType.CATCHUP }}")
        return endDate != null
                && endDate > TimeProvider.now().minusSeconds(item.catchupOffsetInSeconds)
                //TODO: below is attempted fix of restriction in scope of PRJ1010YOU-2634.
                //event CATCHUP restriction seems incorrectly redistributed via channel restrictions to multiple events.
                && item.restrictions.find { it.type == SmartRestrictionType.CATCHUP } == null
    }

    override fun getCatchupMediaStreamReference(
        detail: SmartMediaDetail,
        channel: SmartMediaItem?
    ): SmartVideoStreamReference? {
        return detail.editions.firstOrNull()?.streams?.firstOrNull() as? SmartVideoStreamReference
    }

    override fun getSmartMediaUriParamsFromReference(reference: SmartMediaReference): Uri? =
            Uri.parse(reference.toSmartDataValue().toJsonTree().toString())

    override fun getSmartMediaReferenceFromUri(uri: Uri): SmartMediaReference? =
            SmartVideoReferenceFactory().toSmartMediaReference(SmartDataValue.fromJson(Gson().fromJson(uri.toString(), JsonElement::class.java)))

    override fun getSmartMediaReferenceForId(id: String): SmartMediaReference = SmartVideoMediaReference(id, SmartMediaType.UNKNOWN)

    override suspend fun getLauncherPage(
        smartApi: FlowSmartApi?,
        pageReference: SmartPageReference
    ): SmartPage? {
        val page = smartApi?.getPage(pageReference)?.last()
        if (pageReference is SmartVideoPageReference) {
            when (pageReference.type) {
                SmartVideoPageReference.Type.EPG -> { /* nothing */ }
                SmartVideoPageReference.Type.GRID -> { /* no change needed */ }
                SmartVideoPageReference.Type.MAIN -> {
                    // insert recommended apps
                    val sections = page?.sections?.toMutableList() ?: emptyList<SmartPageSection>().toMutableList()
                    if (sections.isNotEmpty() && sections[0].playlistReference == SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.ON_NOW_PROGRAMS)){
                        sections.add(1, HardcodedPageMapper.convertRecommendedApps()) // Recommended Apps row positioned under On Now row
                    } else {
                        sections.add(0, HardcodedPageMapper.convertRecommendedApps()) // Recommended Apps row positioned first in case On Now is missing
                    }
                    page?.sections = sections//.takeLast(3)
                }
            }
        }
        return page
    }

    override suspend fun getSubscriptions(
        smartApi: FlowSmartApi,
        reference: SmartMediaReference
    ) = emptyList<SmartSubscription>()

    override suspend fun subscribeProduct(
        smartApi: FlowSmartApi,
        productReference: SmartSubscriptionReference?,
        isAutoRenew: Boolean?,
        subscriptionProduct: SmartSubscription?
    ) : SmartPurchase = SmartPurchase(-1, SmartPurchaseState.FAILED)

    override fun getMediaRestrictionViewModel(): MediaRestrictionIconsViewModel = KijkwijzerIconsViewModel()

    override fun getReplacedRecommendedApps(context: Context): List<Pair<String, String>> {
        return listOf(Pair(context.getString(R.string.videoland_package_name), context.getString(R.string.videoland_v2_package_name)))
    }

    override fun getTimezoneID(): String? = TIMEZONE_ID

    override fun getHomepageChannel_SmartPageSection_AllChannels(): SmartPageSection? = SmartPageSection(
        id = "all_channels",
        label = "All Channels",
        playlistReference = SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.ALL_CHANNELS)
    ).apply { sectionStyle = SmartVideoPlaylistReference.Type.ALL_CHANNELS.toString() }

    override fun getHomepageChannel_SmartPageSection_OnNow(): SmartPageSection? = SmartPageSection(
        id = "on_now",
        label = "On Now",
        playlistReference = SmartVideoPlaylistReference(SmartVideoPlaylistReference.Type.ON_NOW_PROGRAMS)
    ).apply { sectionStyle = SmartVideoPlaylistReference.Type.ON_NOW_PROGRAMS.toString() }

    override fun getAllChannelsSectionStyle(): String? = SmartVideoPlaylistReference.Type.ALL_CHANNELS.toString()

    override fun getDrmCallback(drmCallback: MediaDrmCallback?, drmToken: String, streamUrl: String): MediaDrmCallback {
        val assetConfiguration = AssetConfiguration.Builder()
            .tokenWith(drmToken)
            .kidProviderWith(
                HttpKidSource(URL(streamUrl))
            )
            .build()
        return WidevineCallback(assetConfiguration)
    }

    override fun getDrmScheme(): UUID {
        return vudrm.widevineDRMSchemeUUID
    }

    override fun getSearchSorting(genre: SearchGenre): SmartPlaylistSorting? {
        val sorting = when (genre) {
            //for some reason PUB_DATE_ fail when used, server returns error (invalid sort) even if this sort is in documentation
//            SearchGenre.LIVE_TV -> SmartVideoPlaylistSortingReference.Type.PUB_DATE_ASC
//            SearchGenre.CATCHUP -> SmartVideoPlaylistSortingReference.Type.PUB_DATE_DESC
//            SearchGenre.RECORDINGS -> SmartVideoPlaylistSortingReference.Type.PUB_DATE_DESC
            SearchGenre.LIVE_TV,
            SearchGenre.CATCHUP,
            SearchGenre.RECORDINGS -> SmartVideoPlaylistSortingReference.Type.NONE
            SearchGenre.CHANNELS -> SmartVideoPlaylistSortingReference.Type.CHANNEL_NAME_ASC
        }
        return SmartPlaylistSorting(
            reference = SmartVideoPlaylistSortingReference(sorting),
            label = sorting.label
        )
    }

    override fun getSearchFilter(genre: SearchGenre): SmartPlaylistFilterReference? = when (genre) {
        SearchGenre.LIVE_TV -> SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.LIVE_TV
        SearchGenre.CATCHUP -> SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CATCHUP
        SearchGenre.RECORDINGS -> SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.RECORDINGS
        SearchGenre.CHANNELS -> SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CHANNELS
        else -> null
    }?.let(::SmartVideoPlaylistFilterReferenceContent)

    override fun isRecordingRow(section: SmartPageSection): Boolean =
        (section.playlistReference as? SmartVideoPlaylistReference)?.type == SmartVideoPlaylistReference.Type.RECORDINGS

    override fun populateDatabase(smartApi: FlowSmartApi?): Flow<SSTResult<Int>>? {
        return (smartApi as? SmartVideoFlowSmartApi)?.populateDatabase()

        //debug flow
//        return flow {
//            (0..9).forEach {
//                Log.d("Flow", "populateDatabase emit: $it")
//                emit(SSTResult.loading(it))
//                delay(1000)
//                if(it == 5) throw IllegalStateException("test")
//            }
//            Log.d("Flow", "populateDatabase emit: 100")
//            emit(SSTResult.success(100))
//        }
    }

    override fun getActionForKeyEvent(event: KeyEvent): KeyEventAction {
        return when (event.keyCode){
            KEYCODE_BACK,
            KEYCODE_BUTTON_B -> KeyEventAction.BACK
            KEYCODE_MEDIA_STOP -> KeyEventAction.STOP
            KEYCODE_MEDIA_PLAY_PAUSE -> KeyEventAction.PLAY_PAUSE
            KEYCODE_MEDIA_REWIND -> KeyEventAction.RW
            KEYCODE_MEDIA_FAST_FORWARD -> KeyEventAction.FF
            KEYCODE_CHANNEL_DOWN -> KeyEventAction.CHANNEL_DOWN
            KEYCODE_CHANNEL_UP -> KeyEventAction.CHANNEL_UP
            KEYCODE_DPAD_LEFT -> KeyEventAction.DPAD_LEFT
            KEYCODE_DPAD_RIGHT -> KeyEventAction.DPAD_RIGHT
            KEYCODE_DPAD_DOWN -> KeyEventAction.DPAD_DOWN
            KEYCODE_DPAD_UP -> KeyEventAction.DPAD_UP
            KEYCODE_DPAD_CENTER -> KeyEventAction.DPAD_CENTER
            in KEYCODE_NUMPAD_0..KEYCODE_NUMPAD_9,
            in KEYCODE_0..KEYCODE_9 -> KeyEventAction.NUMBER
            KEYCODE_LAST_CHANNEL -> KeyEventAction.LAST
            KEYCODE_PROG_GREEN -> KeyEventAction.STARTOVER
            KEYCODE_INFO -> KeyEventAction.INFO
            KEYCODE_MEDIA_RECORD -> KeyEventAction.RECORD
            KEYCODE_CAPTIONS -> KeyEventAction.AUDIO_SUBTITLES
            else -> return KeyEventAction.NO_ACTION
        }
    }

    override fun pickBasedOnFlavor(smartImages: SmartImages?, width: Int, height: Int, differenceProperty: SmartImages.DifferenceProperty): String? {
        return smartImages?.pick(width, height, differenceProperty)?.let {
            "$it?w=$width&h=$height"
        }
    }

    override fun shouldPostBookmark(event: SmartPlayerEvent, startBookmarkSent: Boolean, isLive: Boolean?): Boolean {
        return super.shouldPostBookmark(event, startBookmarkSent, isLive) && isLive != true
    }

    override val restartAfterStandby = true

    override val shouldDisableHorizontalZapp = false
    override val shouldDisableVerticalZapp = false
    override val minimalDelayForHorizontalRepeat = 0L
    override val minimalDelayForVerticalRepeat = 500L
    override val shouldStopGoToLiveInPlayer = true
    override val shouldShowSubtitlesInfoInPlayer = true
    override val shouldEpgStoreFocusOnDetailOpen: Boolean = true
    override val startOverIconDisplayRule = StartOverIconDisplayRule.CATCHUP_BASED
    override val glideCacheExpirationInterval: Long = System.currentTimeMillis() / (com.twentyfouri.tvlauncher.common.Flavor().epgExpirationDays * MILLIS_PER_DAY)
    override val handleInPlayerMessageCatchupNotAvailableYet = true
    override val adsDurationBeforeRecordingMillis: Int = 300000 // 5 minutes
    override val adsDurationAfterRecordingMillis: Int = 300000 // 5 minutes

    override fun getFilterType(filterReference: SmartPlaylistFilterReference): FilterViewModel.Type =
        when (filterReference) {
            is SmartVideoPlaylistFilterReferenceChannel -> FilterViewModel.Type.CHANNEL
            is SmartVideoPlaylistFilterReferenceDate -> FilterViewModel.Type.DATE
            is SmartVideoPlaylistFilterReferenceInitial -> FilterViewModel.Type.INITIAL
            is SmartVideoPlaylistFilterReferenceContent -> FilterViewModel.Type.CONTENT
            else -> FilterViewModel.Type.NONE
        }

    override fun getFilterDate(filterReference: SmartPlaylistFilterReference?): Int? {
        return (filterReference as? SmartVideoPlaylistFilterReferenceDate)?.dayDiff?.toInt()
    }

    override fun getFilterContent(filterReference: SmartPlaylistFilterReference?): SmartMediaType {
        return when ((filterReference as? SmartVideoPlaylistFilterReferenceContent)?.content) {
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.LIVE_TV -> SmartMediaType.LIVE_VIDEO
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CATCHUP -> SmartMediaType.LIVE_EVENT
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CHANNELS -> SmartMediaType.LIVE_CHANNEL
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.RECORDINGS -> SmartMediaType.RECORDING
            else -> SmartMediaType.UNKNOWN
        }
    }

    override fun getFilterContentTranslation(filterReference: SmartPlaylistFilterReference?, resourceRepository: ResourceRepository): String? {
        return when ((filterReference as? SmartVideoPlaylistFilterReferenceContent)?.content) {
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.LIVE_TV -> resourceRepository.getString(R.string.filter_content_live)
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CATCHUP -> resourceRepository.getString(R.string.filter_content_catchup)
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.CHANNELS -> resourceRepository.getString(R.string.filter_content_channel)
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.RECORDINGS -> resourceRepository.getString(R.string.filter_content_recording)
            SmartVideoPlaylistFilterReferenceContent.SmartVideoContent.LAST_WATCHED -> resourceRepository.getString(R.string.filter_content_last_watched)
            else -> null
        }
    }

    override fun applyAdditionalFilters(selectedFilterRefs: List<SmartPlaylistFilterReference>): List<SmartPlaylistFilterReference> {
        return selectedFilterRefs.toMutableList().apply { add(SmartVideoPlaylistFilterReferenceGenre("",GenreToFilterAdultChannels)) }
    }

    override fun getFormattedDateForGrid(originalDate: String): String = originalDate.replace(".","")


    class EPGConfigValuesYoufone: EpgFragment.EPGConfigValues() {
        override val SCROLL_PAGE_SIZE = 5
        override val INACTIVITY_REFRESH_MS = 1000 * 60 * 30
        override val HORIZONTAL_PAGE_SIZE_MS = 1000 * 60 * 60 * 2
        override val VERTICAL_PAGE_SIZE_CHANNEL_COUNT = 8
        override val EPG_DAYS_INTO_PAST = com.twentyfouri.tvlauncher.common.Flavor.EPG_DAYS_INTO_PAST
        override val EPG_DAYS_INTO_FUTURE = com.twentyfouri.tvlauncher.common.Flavor.EPG_DAYS_INTO_FUTURE
        override val MILLISECONDS_IN_DAY = 24 * 60 * 60 * 1000
        override val REQUEST_FOCUS = "REQUEST_FOCUS"
    }
}