package com.twentyfouri.tvlauncher.widgets

import android.content.Context
import android.graphics.Rect
import android.util.AttributeSet
import android.util.Log
import android.view.View
import android.view.ViewGroup
import android.widget.FrameLayout
import com.twentyfouri.tvlauncher.R
import timber.log.Timber
import java.util.ArrayList

open class FocusHandleFrameLayout : FrameLayout {

    var blockFocusDirectionDown = false
    var blockFocusDirectionUp = false
    var blockFocusDirectionLeft = false
    var blockFocusDirectionRight = false
    var focusCatch = false
    var focusRemember = false

    private var lastFocusedView: View? = null
    private var lastRequestedFocusedView: View? = null

    var focusBlockListener: FocusBlockListener? = null
    var focusRememberListener: FocusRememberListener? = null

    var disablePostCallingFocus = false

    interface FocusBlockListener {
        fun onFocusBlocked(v: View, direction: Int)
    }

    interface FocusRememberListener {
        fun provideChildToBeFocused(
            child: View?,
            focused: View?,
            lastFocusedView: View?
        ) : View?
    }

    constructor(context: Context) : super(context) { init(attrs = null) }
    constructor(context: Context, attrs: AttributeSet?) : super(context, attrs) { init(attrs) }
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int) : super(context, attrs, defStyleAttr) { init(attrs) }
    constructor(context: Context, attrs: AttributeSet?, defStyleAttr: Int, defStyleRes: Int) : super(
        context,
        attrs,
        defStyleAttr,
        defStyleRes
    ) { init(attrs) }

    private fun init(attrs: AttributeSet?) {
        val a = context.obtainStyledAttributes(attrs, R.styleable.FocusHandleFrameLayout)
        for (index in 0 until a.indexCount) {
            val attr = a.getIndex(index)
            when (attr) {
                R.styleable.FocusHandleFrameLayout_blockFocusDirectionDown -> blockFocusDirectionDown = a.getBoolean(attr, false)
                R.styleable.FocusHandleFrameLayout_blockFocusDirectionUp -> blockFocusDirectionUp = a.getBoolean(attr, false)
                R.styleable.FocusHandleFrameLayout_blockFocusDirectionLeft -> blockFocusDirectionLeft = a.getBoolean(attr, false)
                R.styleable.FocusHandleFrameLayout_blockFocusDirectionRight -> blockFocusDirectionRight = a.getBoolean(attr, false)
                R.styleable.FocusHandleFrameLayout_focusCatch -> focusCatch = a.getBoolean(attr, false)
                R.styleable.FocusHandleFrameLayout_focusRemember -> focusRemember = a.getBoolean(attr, false)
            }
        }
        a.recycle()
    }

    /*
    Focus Block Handling
     */
    override fun focusSearch(focused: View?, direction: Int): View? {
        val isPreviousChildOfRecycler = isViewChildOfOurs(this, focused)
        //    	View newPotentialFocus = super.focusSearch(focused, direction);
        // iterate until we find correct next focus inside our layout or until we hit edge (not our view)
        val newPotentialFocus = findFirstVisibleFocusDest(focused, direction)
        val isChildOfRecycler = isViewChildOfOurs(this, newPotentialFocus)
        Timber.tag(VIEW_LOG_TAG).d("focusSearch newPotentialFocus = $newPotentialFocus")

        if (isPreviousChildOfRecycler) {
            if (direction == View.FOCUS_LEFT && !isChildOfRecycler && blockFocusDirectionLeft) {
                focusBlockListener?.onFocusBlocked(this, View.FOCUS_LEFT)
                return null
            }
            if (direction == View.FOCUS_RIGHT && !isChildOfRecycler && blockFocusDirectionRight) {
                focusBlockListener?.onFocusBlocked(this, View.FOCUS_RIGHT)
                return null
            }
            if (direction == View.FOCUS_UP && !isChildOfRecycler && blockFocusDirectionUp) {
                focusBlockListener?.onFocusBlocked(this, View.FOCUS_UP)
                return null
            }
            if (direction == View.FOCUS_DOWN && !isChildOfRecycler && blockFocusDirectionDown) {
                focusBlockListener?.onFocusBlocked(this, View.FOCUS_DOWN)
                return null
            }
        }

        if (focusRemember) {
            // if newPotentialFocus is not child of ours, then remember it for autofocus on focus gain
            if (!isViewChildOfOurs(this, newPotentialFocus)) {
                lastFocusedView = focused
            }
        }

        return newPotentialFocus
    }

    private fun isViewChildOfOurs(parent: ViewGroup, v: View?): Boolean {
        val viewCount = parent.childCount
        for (index in 0 until viewCount) {
            val child = parent.getChildAt(index)
            if (v === child)
                return true
            if (child is ViewGroup) {
                val resultNested = isViewChildOfOurs(child, v)
                if (resultNested)
                    return true
            }
        }
        return false
    }

    private fun findFirstVisibleFocusDest(focused: View?, direction: Int): View? {
        var lastFoundFocusDest = focused
        do {
            val newPotentialFocus = super.focusSearch(lastFoundFocusDest, direction)
            val isChildOfRecycler = isViewChildOfOurs(this, newPotentialFocus)
            if (!isChildOfRecycler)
                return newPotentialFocus
            if (newPotentialFocus === lastFoundFocusDest)
                return lastFoundFocusDest
            lastFoundFocusDest = newPotentialFocus
        } while (lastFoundFocusDest?.isShown != true)
        return lastFoundFocusDest
    }
    /*
    End of Focus Block Handling
     */


    /*
    Focus Catch Handling
     */
    override fun requestFocus(direction: Int, previouslyFocusedRect: Rect?): Boolean {
        if (focusCatch) {
            Timber.tag(VIEW_LOG_TAG).d("requestFocus direction = $direction")
            val focusInChild = onRequestFocusInDescendants(direction, previouslyFocusedRect)
            Timber.tag(VIEW_LOG_TAG).d("requestFocus focus found in children = $focusInChild")
            return if (focusInChild) true else super.requestFocus(direction, previouslyFocusedRect)
        }

        return super.requestFocus(direction, previouslyFocusedRect)
    }

    /*
    Focus Remember Handling
     */
    override fun clearFocus() {
        if (focusRemember) {
            // this is dirty hack, because focus can be lost without focusSearch being called on sibling view
            if (lastRequestedFocusedView != null)
                lastFocusedView = lastRequestedFocusedView
        }
        super.clearFocus()
    }

    override fun requestChildFocus(child: View?, focused: View?) {
        if (focusRemember) {
            // check if we had some focused view before
            if (lastFocusedView != null || findFocus() == null) {
                val localFocusReqView = focusRememberListener?.provideChildToBeFocused(child, focused, lastFocusedView) ?: lastFocusedView
                lastFocusedView = null
                // remember last requested focus on view under this layout tree
                lastRequestedFocusedView = localFocusReqView
                // This has to be called later, because of default focus changes invoked by super.requestChildFocus ->
                // this was causing onFocusChanged on FocusableContainer, what called requestFocus on its child
                if (!disablePostCallingFocus) post { localFocusReqView?.requestFocus() }
                // let layout to finish his focus set things
                super.requestChildFocus(child, focused)
            } else {
                // remember last requested focus on view under this layout tree
                lastRequestedFocusedView = focused
                super.requestChildFocus(child, focused)
            }
        }
        else {
            super.requestChildFocus(child, focused)
        }
    }

    override fun addFocusables(views: ArrayList<View>?, direction: Int) {
        if (descendantFocusability != ViewGroup.FOCUS_BLOCK_DESCENDANTS) super.addFocusables(views, direction)
    }

    override fun addFocusables(views: ArrayList<View>?, direction: Int, focusableMode: Int) {
        if (descendantFocusability != ViewGroup.FOCUS_BLOCK_DESCENDANTS) super.addFocusables(views, direction, focusableMode)
    }

}