package com.twentyfouri.tvlauncher.viewmodels

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import androidx.lifecycle.ViewModel
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.recording.SmartRecordingStatus
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.data.ColumnItem
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.utils.ExpirableGlideImageSpecification

class RecordingItemViewModel(
    resourceRepository: ResourceRepository,
    dateTimeRepository: DateTimeRepository
) : ViewModel() {
    private val columnItem = MutableLiveData<ColumnItem>()
    private val _onItemClicked = MutableLiveData<ColumnItem?>()

    val imageSpecification: LiveData<ImageSpecification>
    val recordingTitle: LiveData<String>
    val recordingChannel: LiveData<String>
    val recordingTime: LiveData<String>
    val recordingDate: LiveData<String>
    val onItemClicked: LiveData<ColumnItem?> = _onItemClicked

    init {
        imageSpecification = Transformations.map(columnItem) { it.recording?.getImageSpecification(resourceRepository) }
        recordingTitle = Transformations.map(columnItem) { it.recording?.title }
        recordingChannel = Transformations.map(columnItem) { it.recording?.channelName }
        recordingTime = Transformations.map(columnItem) {
            when {
                it.recordingEpisodes.isNullOrEmpty() -> it.recording?.getDurationOrStartTime(dateTimeRepository) ?: ""
                else -> resourceRepository.getString(R.string.recordings_episodes, it.recordingEpisodes!!.size.toString())
            }
        }
        recordingDate = Transformations.map(columnItem) { it.recording?.getDate(dateTimeRepository) }
    }

    private fun SmartMediaItem.getImageSpecification(resourceRepository: ResourceRepository): ImageSpecification {
        val foundUrl = Flavor().pickBasedOnFlavor(images,
            resourceRepository.getDimensionPixelSize(R.dimen.recording_image_width),
            resourceRepository.getDimensionPixelSize(R.dimen.recording_image_height),
            SmartImages.UNRESTRICTED
        )
        return if (foundUrl.isNullOrEmpty()) EmptyImageSpecification()
        else ExpirableGlideImageSpecification(foundUrl, R.drawable.default_image_small).apply {
            //Log.e("GlideImageSpecification", "url = ${toString()}")
        }
    }

    private fun SmartMediaItem.getDurationOrStartTime(dateTimeRepository: DateTimeRepository) = when(recordingData?.recordingStatus) {
        SmartRecordingStatus.WAIT_RECORD -> dateTimeRepository.formatClockTime(startDate?.millis ?: TimeProvider.now().millis)
        SmartRecordingStatus.SUCCESS -> dateTimeRepository.formatDuration(startDate, endDate)
        else -> dateTimeRepository.formatClockTime(TimeProvider.now().millis)
    }

    private fun SmartMediaItem.getDate(dateTimeRepository: DateTimeRepository) = dateTimeRepository.formatDateTime(
        startDate?.plusSeconds(recordingData?.beginOffsetInSeconds ?: 0),
        endDate?.plusSeconds(recordingData?.endOffsetInSeconds ?: 0)
    )

    fun setColumnItem(item: ColumnItem) { columnItem.value = item }

    fun onClick() {
        _onItemClicked.value = columnItem.value }

    fun resetClickedItem(){
        _onItemClicked.value = null
    }
}