package com.twentyfouri.tvlauncher.viewmodels

import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Transformations
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.media.SmartAgeRating
import com.twentyfouri.smartmodel.model.media.SmartRestrictionType
import com.twentyfouri.smartmodel.model.media.SmartSeekingRuleType
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.utils.CombinationTransformations

class PlayerUIVodViewModel(
    private val dateTimeRepository: DateTimeRepository,
    metadataRepository: MetadataRepository,
    resourceRepository: ResourceRepository,
    youboraAnalytics: YouboraAnalytics
) : PlayerUIViewModel(metadataRepository, dateTimeRepository, resourceRepository, youboraAnalytics) {

    //observed from View
    val barMaxTimeString: LiveData<String>
    val barCurrentTimeString: LiveData<String>
    val catchupTitle: LiveData<String>

    //other produced LiveData
    val ageRatings: LiveData<MutableList<SmartAgeRating>>
    @Suppress("JoinDeclarationAndAssignment")
    private val barMaxTime: LiveData<Long>
    private val barCurrentTime: LiveData<Long>

    private var internalFastForwardAllowed: Boolean = true

    init {
        barMaxTime = Transformations.map(playerDuration) { if (it>0) it else 0 }
        barCurrentTime = CombinationTransformations.combineNonNullable(playerDuration, playerPosition) { aDuration, aPosition -> if (aDuration > 0) aPosition else 0 }
        barMaxTimeString = Transformations.map(barMaxTime) { dateTimeRepository.formatTime(it) }
        barCurrentTimeString = Transformations.map(barCurrentTime) { dateTimeRepository.formatTime(it) }
        catchupTitle = Transformations.map(detail) {
            internalFastForwardAllowed = true
            it.title
        }
        ageRatings = Transformations.map(detail) { it.ageRatings }
    }

    //region PlayerUIViewModel overrides

    override fun seekTick(streamDuration: Long?, seekIncrement: Int, seekMultiplier: Int): Int {
        val shouldAllowFiveMinuteFastForward = detail.value?.restrictions?.find { it.type == SmartRestrictionType.FAST_FORWARD } != null
        var newSeekMultiplier = seekMultiplier
        barProgress.value?.let { primaryProgress ->
            var currentPos = primaryProgress
            currentPos += seekIncrement
            if (currentPos < 0) {
                currentPos = 0
                if (seekMultiplier < 0) newSeekMultiplier = 0
            }
            streamDuration?.let {
                if (currentPos > it) {
                    currentPos = it.toInt()
                    if (seekMultiplier > 0) newSeekMultiplier = 0
                }
            }
            if (shouldAllowFiveMinuteFastForward && seekIncrement > 0 && currentPos > FF_MINUTES_ALLOWED_FOR_RECORDING) { //5 minutes ff is allowed for recorded items with FAST_FORWARD restriction
                newSeekMultiplier = 0
                internalFastForwardAllowed = false // blocking FF after 5 minutes
            }
            else
                playerPosition.postValue(currentPos.toLong())

            if (currentPos <= FF_MINUTES_ALLOWED_FOR_RECORDING)
                internalFastForwardAllowed = true // allowing FF again for the first 5 minutes
        }
        return newSeekMultiplier
    }

    override fun seekToSelectedPos(player: SmartPlayer?, isInTrick: Boolean) {
        barProgress.value?.also { player?.seekTo(it.toLong()) }
    }

    override fun getSeekingRuleTypeInternal() = SmartSeekingRuleType.TIMESHIFT

    override fun isFastForwardAllowed(): Boolean = internalFastForwardAllowed

    override fun getChannelReferenceByNumber(number: String): SmartMediaReference? {
        return try {
            getChannelByNumber(number.toInt())?.reference
        } catch (e: NumberFormatException) {
            null
        }
    }

    override fun getClosestChannelReferenceByNumber(number: String): SmartMediaReference? {
        return try {
            getClosestChannelByNumber(number.toInt(), channels.value)?.reference
        } catch (e: NumberFormatException) {
            null
        }
    }

    //endregion
}