package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import androidx.lifecycle.*
import androidx.lifecycle.Observer
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.androidcore.utils.ResourceImageSpecification
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.error.PlaybackRestrictedException
import com.twentyfouri.smartmodel.model.media.*
import com.twentyfouri.smartmodel.model.watchlist.SmartContinueWatchingItem
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.RestrictionChecker
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_ERROR_LOG
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.MetadataRepository
import com.twentyfouri.tvlauncher.utils.CleanableMediatorLiveData
import com.twentyfouri.tvlauncher.utils.CombinationTransformations
import kotlinx.coroutines.*
import timber.log.Timber
import java.util.*
import kotlin.concurrent.fixedRateTimer
import kotlin.math.abs

abstract class PlayerUIViewModel(
        private val metadataRepository: MetadataRepository,
        private val dateTimeRepository: DateTimeRepository,
        protected val resourceRepository: ResourceRepository,
        private val youboraAnalytics: YouboraAnalytics
) : ViewModel() {

    companion object {
        internal const val CURRENT_TIME_DELAY = 1000L
        internal const val CURRENT_TIME_PERIOD = 1000L
        internal const val BAR_MAX_DEFAULT = 100
        internal const val FF_MINUTES_ALLOWED_FOR_RECORDING = 300000 // 5 minutes
    }

    private enum class PlayPauseIconType {
        PAUSE_FROM_SEEK, PAUSE_FROM_PLAYER, PAUSE_FROM_USER, PLAY_FROM_SEEK, PLAY_FROM_PLAYER, PLAY_FROM_USER, FF, RW
    }

    var errorState = false

    //observed from View
    val subscriptionChannel = MutableLiveData<SmartMediaItem?>() //if non-null value is set, subscription flow is triggered
    val playFailedVisibility = MutableLiveData<Int>()
    val playFailedText = MutableLiveData<String>(resourceRepository.getString(R.string.play_failed_channel))
    val streamPausedVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    val playRestrictedVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    val seekText = MutableLiveData<String>()
    val currentTimeString: LiveData<String>
    val barMax: LiveData<Int>
    val barProgress: LiveData<Int>
    val barSecondaryProgress: LiveData<Int>
    val playingEvent = CleanableMediatorLiveData<SmartMediaItem?>()
    var loadingEvents: Boolean = false
    val isCatchupSubscribed = MutableLiveData<Boolean>().apply { value = true }
    val playingEventParentalLD: LiveData<SmartMediaItem?>
    val playPauseIcon: LiveData<ImageSpecification>

    //input LiveData
    protected lateinit var channels: LiveData<List<SmartMediaItem>>
    protected val detail = MutableLiveData<SmartMediaDetail>()
    protected val currentTimeMillis = MutableLiveData<Long>()
    protected val playerPosition = MutableLiveData<Long>()
    protected val playerDuration = MutableLiveData<Long>()
    protected val playerBufferedPosition = MutableLiveData<Long>()
    private val playPauseIconType = MutableLiveData<PlayPauseIconType>()

    private var timeTick: Timer? = null
    protected var lastPlayedChannelNumber: Int? = null

    init {
        clearData()
        currentTimeString = Transformations.map(currentTimeMillis) { dateTimeRepository.formatClockTime(it) }
        barMax = Transformations.map(playerDuration) { aDuration -> if (aDuration>0) aDuration.toInt() else BAR_MAX_DEFAULT }
        barProgress = CombinationTransformations.combineNonNullable(playerDuration, playerPosition) { aDuration, aPosition -> if (aDuration > 0) aPosition.toInt() else 0 }
        barSecondaryProgress = CombinationTransformations.combineNonNullable(playerDuration, playerBufferedPosition) { aDuration, aPosition -> if (aDuration > 0) aPosition.toInt() else 0 }
        playingEventParentalLD = Transformations.map(playingEvent){ it }
        playPauseIconType.value = PlayPauseIconType.PLAY_FROM_PLAYER
        playPauseIcon = Transformations.map(playPauseIconType) {
            ResourceImageSpecification(
                when (it) {
                    PlayPauseIconType.PAUSE_FROM_PLAYER,
                    PlayPauseIconType.PAUSE_FROM_SEEK,
                    PlayPauseIconType.PAUSE_FROM_USER -> R.drawable.player_controls_pause
                    PlayPauseIconType.FF -> R.drawable.player_controls_ff
                    PlayPauseIconType.RW -> R.drawable.player_controls_rw
                    else -> R.drawable.player_controls_play
                }
            )
        }
    }

    //can be override
    open fun resetPauseTime() = Unit
    open fun setPauseTime() = Unit
    open fun onPlayerActionStop() = Unit
    open fun startSeekingPrepare() = Unit
    open fun getNextChannelReference(): SmartMediaReference? = null
    open fun getNextChannelIdTo(channelReference: SmartMediaReference): String? = null
    open fun getPreviousChannelReference(): SmartMediaReference? = null
    open fun getPreviousChannelIdTo(channelReference: SmartMediaReference): String? = null
    open fun softZapNextChannel() = Unit
    open fun softZapPreviousChannel() = Unit
    open fun softZapNextEvent() = Unit
    open fun softZapPreviousEvent() = Unit
    open fun getSoftZapMediaReference(): SmartMediaReference? = null
    open fun cancelSoftZap() = Unit
    @Suppress("SpellCheckingInspection")
    open fun softZaping(): Boolean = false
    open fun getChannelReferenceByNumber(number: String): SmartMediaReference? = null
    open fun getClosestChannelReferenceByNumber(number: String): SmartMediaReference? = null
    open fun getLastChannelReference(fromCatchup: Boolean): SmartMediaReference? = null
    open fun getLastPlayedEventReference(): SmartMediaReference? = null
    open fun cancelSeeking() = Unit
    open fun getPlayingEventStart(): Long = 0
    open fun isTimeShifted(): Boolean = false
    open fun setPlayingChannel(channel: SmartMediaItem) = Unit
    open fun isFastForwardAllowed(): Boolean = detail.value?.restrictions?.find { it.type == SmartRestrictionType.FAST_FORWARD } == null
    open fun getPlayingEventTitleLD(): LiveData<String>? = null
    open fun isSoftZappedPlayableCatchup(): Boolean = true

    open fun observeMediaStreamLD(
        detail: SmartMediaDetail,
        channel: SmartMediaItem?,
        doBlock: (stream: SmartMediaStream) -> Unit,
        catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean,
        lifecycleOwner: LifecycleOwner,
        isPlayingCatchup: Boolean = false
    ) {

        val newCatchBlock: suspend CoroutineScope.(e: Exception) -> Boolean = newCatchBlock@{ e: Exception ->
            GlobalScope.launch(Dispatchers.IO) { youboraAnalytics.reportPlayerError(e) }
            if (e is PlaybackRestrictedException) {
                errorState = true
                RestrictionChecker.handlePlaybackRestrictions(
                    exception = e,
                    playingChannel = channel,
                    isPlayingCatchup = isPlayingCatchup,
                    playFailedVisibility = playFailedVisibility,
                    playFailedText = playFailedText,
                    isCatchupSubscribed = isCatchupSubscribed,
                    subscriptionChannel = subscriptionChannel,
                    resourceRepository = resourceRepository
                )
                return@newCatchBlock true
            }
            return@newCatchBlock catchBlock(this, e)
        }

        playFailedVisibility.postValue(View.GONE)
        playFailedText.postValue(resourceRepository.getString(R.string.play_failed_channel)) //reset default error text
        Flavor().getCatchupMediaStreamReference(detail, channel)?.let {
            val mediaStreamLD = metadataRepository.getMediaStreamLD(
                it,
                newCatchBlock
            )
            mediaStreamLD.observe(lifecycleOwner, Observer { stream ->
                doBlock.invoke(stream)
                mediaStreamLD.removeObservers(lifecycleOwner)
            })
        } ?: GlobalScope.launch {
            Timber.tag(TAG_PLAYER_ERROR_LOG).e(resourceRepository.getString(R.string.missing_stream_reference))
            withContext(Dispatchers.IO) { youboraAnalytics.plugin.fireError(resourceRepository.getString(R.string.missing_stream_reference), resourceRepository.getString(R.string.missing_reference_code),"") }
            catchBlock.invoke(this, IllegalArgumentException(resourceRepository.getString(R.string.missing_stream_reference)))
        }
    }

    open fun onPlayerTimeUpdate(position: Long, duration: Long) {
        playerDuration.value = duration
        playerPosition.value = position
    }

    //must be override
    abstract fun seekTick(streamDuration: Long?, seekIncrement: Int, seekMultiplier: Int): Int
    abstract fun seekToSelectedPos(player: SmartPlayer?, isInTrick: Boolean)
    abstract fun getSeekingRuleTypeInternal(): SmartSeekingRuleType?

    fun getSeekingRule() = getSeekingRuleTypeInternal() ?: SmartSeekingRuleType.LIVE_ONLY

    fun setDetail(detail: SmartMediaDetail) { this.detail.value = detail }

    fun sendPlayerEvent(event: SmartPlayerEvent, bookmark: SmartContinueWatchingItem?) {
        metadataRepository.sendPlayerEvent(event, bookmark)
    }

    fun stopTimer() {
        clearData()
        timeTick?.cancel()
    }

    fun startTimer() {
        stopTimer()
        timeTick = fixedRateTimer(null, false, CURRENT_TIME_DELAY, CURRENT_TIME_PERIOD) { currentTimeMillis.postValue(TimeProvider.now().millis) }
    }

    private fun clearData() {
        seekText.value = ""
        playFailedVisibility.value = View.GONE
        playerPosition.value = 0
        playerDuration.value = 0
        playerBufferedPosition.value = 0
    }

    fun resetTimeMillis() {
        currentTimeMillis.value = -1L
    }

    fun getClosestChannelByNumber(channelNumber: Int, channels: List<SmartMediaItem>?): SmartMediaItem? {
        val channelNumbers = channels?.map { it.channelNumber } ?: return null
        var number = channelNumbers.minByOrNull { abs(channelNumber - it) } //finding closest value
        if (number != null && number < channelNumber) { //closest value must be higher than the original
            val index = channelNumbers.indexOf(number)
            number = channelNumbers.getOrNull(index + 1)
        }
        if (number == null) number = channelNumbers[0]
        return channels.find { it.channelNumber == number }
    }

    protected fun getChannelByNumber(channelNumber: Int): SmartMediaItem? {
        return channels.value?.find { it.channelNumber == channelNumber }
    }

    internal fun applySeekMultiplier(multiplier: Int) {
        playPauseIconType.value = when {
            multiplier < 0 -> PlayPauseIconType.RW
            multiplier > 0 -> PlayPauseIconType.FF
            Flavor().showPauseIconOnlyIfPausedByUser -> PlayPauseIconType.PLAY_FROM_SEEK
            else -> PlayPauseIconType.PAUSE_FROM_SEEK
        }
    }

    internal fun applyIsPlaying(isPlaying: Boolean) {
        //the missing "else" branch is intended. I have not found a way how to incorporate inputs both from player and user
        // when I wanted to have consistent reproducable behavior. Thus if the user triggered states should have priority
        // the player triggered states must be completely ignored, or the output gets pretty random
        if (Flavor().showPauseIconOnlyIfPausedByUser.not())
            playPauseIconType.value = if (isPlaying) PlayPauseIconType.PLAY_FROM_PLAYER else PlayPauseIconType.PAUSE_FROM_PLAYER
    }

    internal fun applyManualPause(isPaused: Boolean) {
        playPauseIconType.value = if (isPaused) PlayPauseIconType.PAUSE_FROM_USER else PlayPauseIconType.PLAY_FROM_USER
    }
}