package com.twentyfouri.tvlauncher.viewmodels

import android.util.Log
import android.view.View
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.Observer
import androidx.lifecycle.Transformations
import com.twentyfouri.androidcore.epg.EpgView
import com.twentyfouri.androidcore.epg.model.EpgEvent
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartexoplayer.SmartPlayer
import com.twentyfouri.smartmodel.model.dashboard.SmartImages
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaReference
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaType
import com.twentyfouri.smartmodel.model.error.PlaybackRestrictedException
import com.twentyfouri.smartmodel.model.error.RecordingStorageException
import com.twentyfouri.smartmodel.model.media.*
import com.twentyfouri.smartmodel.model.menu.SmartNavigationAction
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.StartOverIconDisplayRule
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifNull
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.RestrictionChecker
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.*
import com.twentyfouri.tvlauncher.extensions.seekingRuleAllowsCatchup
import com.twentyfouri.tvlauncher.utils.*
import kotlinx.coroutines.*
import org.joda.time.DateTime
import timber.log.Timber
import java.util.*
import kotlin.concurrent.schedule
import kotlin.random.Random.Default.nextInt

/**
 *         SLW            <- LW ->          ELW/CT
 *          |-------------------|--------------|
 *                             LP
 *                 |---------------------------------------|
 *                ES                <- EW ->               EE
 *                 |------------|--------------|
 *                             PP             SP
 *
 * LW = live window (duration)
 * SLW = start of live window
 * ELW = end of live window
 * LP = position in live stream
 * CT = current time (real)
 * EW = event window
 * ES = event start
 * EE = event end
 * PP = primary progress (white)
 * SP = secondary progress (red)
 *
 * ELW, CT and SP are same points on timeline
 * LP and PP are same points on timeline
 *
 * SP = CT - ES
 * PP = CT - LW + LP
 * */

class PlayerUILiveViewModel(
        private val epgRepository: EpgRepository,
        private val metadataRepository: MetadataRepository,
        private val dateTimeRepository: DateTimeRepository,
        private val recordingsRepository: RecordingsRepository,
        resourceRepository: ResourceRepository,
        private val youboraAnalytics: YouboraAnalytics
) : PlayerUIViewModel(metadataRepository, dateTimeRepository, resourceRepository, youboraAnalytics) {

    companion object {
        @Suppress("SpellCheckingInspection")
        private const val STREAM_BEHIND_LIVE_TRESHOLD = 1 * 60 * 1000 //more than one minutes behind live is considered as time-shifted
        private const val CHANNEL_DELAY_TIMER = "delayedChannelStart"
        private const val CHANNEL_DELAY_LENGTH = 200L
        private const val EVENT_SIZE_ADDITION = 60000 //one minute
    }

    private var useOriginalRecordingState = true

    //observed from View
    val playerClockVisibility: LiveData<Int>
    val channelNumberAndTitle: LiveData<String>
    val channelIcon: LiveData<ImageSpecification>
    val channelIconPreLoad = MutableLiveData<ImageSpecification>(null)
    val barTime = MutableLiveData<String>()
    val playingEventTime: LiveData<String>
    val nextPlayingEventTime: LiveData<String>
    val nextPlayingEventRecording: LiveData<Int>
    val playingEventTitle: LiveData<String>
    val playingEventStartOverVisible: LiveData<Int>
    val nextEventStartOverVisible: LiveData<Int>
    val nextPlayingEventTitle: LiveData<String>
    val buttonNavigationVisible: LiveData<Int>
    val buttonStartOverVisible: LiveData<Int>
    val buttonOptionsVisible: LiveData<Int>
    val buttonBackToLiveVisible: LiveData<Int>
    val buttonOpenSubtitlesVisible: LiveData<Int>

    private val channelLogoDesiredWidth = resourceRepository.getPixelSizeFromDimens(R.dimen.player_controls_channel_icon_width)
    private val channelLogoDesiredHeight = resourceRepository.getPixelSizeFromDimens(R.dimen.player_controls_channel_icon_height)

    val channelIcons: LiveData<List<String>>
    val playingEventRecording: MutableLiveData<Int>
    private val playingChannel = MutableLiveData<SmartMediaItem?>(null)
    private val softZappChannel = MutableLiveData<SmartMediaItem?>(null)

    private val currentEvent = CleanableMediatorLiveData<SmartMediaItem?>()
    private val nextEvent = CleanableMediatorLiveData<SmartMediaItem?>()
    private val seekingRuleType: LiveData<SmartSeekingRuleType>
    val ageRatings: LiveData<MutableList<SmartAgeRating>>
    private val recordingState: LiveData<MetadataViewModel.RecordingState?>
    private val userTriggeredRecordingState = MutableLiveData<MetadataViewModel.RecordingState>()
    private val innerDetail: LiveData<SmartMediaDetail>

    private val softZappDirection = MutableLiveData(0)

    //input LiveData
    private val recordings: LiveData<List<SmartMediaItem>?> = recordingsRepository.getRecordingsLD()

    private var barStreamBehindLive = 0L
    private var seeking = false
    private var updateTime = false
    private var pauseTimeMillis = 0L
    private var delayedChannelStartTimer: TimerTask? = null
    private var isPlayingCatchup: Boolean = false

    private val forceUpdateLD = MutableLiveData(1)

    private var epgFocusedEventStore: EpgView.EpgFocusedEventStore? = null

    private val randomMinuteToCacheFutureNextEvent = nextInt(1, 5)
    private var futureNextEvent: LiveData<SmartMediaItem?>? = null
    private var shouldCacheFutureNextEvent = true

    init {
        channels = epgRepository.getAllChannelsLD()

        buttonNavigationVisible = MutableLiveData<Int>().apply { value = View.VISIBLE }

        channelIcons = Transformations.map(channels) { listOfChannels ->
            listOfChannels.map {
                Flavor().pickBasedOnFlavor(Flavor().getImageOfType(it, ImageType.DARK),
                    channelLogoDesiredWidth.toInt(),
                    channelLogoDesiredHeight.toInt(),
                    SmartImages.UNRESTRICTED
                ) ?: ""
            }
        }

        seekingRuleType = CombinationTransformations.combineNullable(playingChannel, playingEvent) { ch, ev ->
            ev?.seekingRuleType ?: ch?.seekingRuleType ?: SmartSeekingRuleType.LIVE_ONLY
        }

        innerDetail = Transformations.switchMap(playingEvent) {
            if (detail.value?.type == SmartMediaType.LIVE_CHANNEL) //when player is opened via channel row, detail is channel and needs to be changed to actual event
                metadataRepository.getMediaDetailLD(it?.reference, it?.channelReference, true) //TODO Is the bookmark needed here?
            else
                detail
        }

        recordingState = CombinationTransformations.combineNullable(currentEvent, recordings) { mediaDetail, recordings ->
            useOriginalRecordingState = true
            when {
                mediaDetail == null -> MetadataViewModel.RecordingState.NOTHING
                recordings == null -> MetadataViewModel.RecordingState.NOTHING
                mediaDetail.isRecordingAllowed().not() -> MetadataViewModel.RecordingState.NOTHING
                else -> mediaDetail.getRecordingState(recordings)
            }
        }

        buttonStartOverVisible = CombinationTransformations.combineHalfNullable(seekingRuleType, currentEvent) { seekingRule, currentEvent ->
            if (Flavor().shouldHandleLongPress()){
                if (currentEvent != null && currentEvent !is DummySmartMediaItem){
                    View.VISIBLE
                } else View.GONE
            } else getVisibilityBasedOnSeekingRuleType(seekingRule)
        }

        buttonOptionsVisible = Transformations.map(seekingRuleType) {
            if (Flavor().shouldHandleLongPress()) View.VISIBLE
            else getVisibilityBasedOnSeekingRuleType(it)
        }

        buttonBackToLiveVisible = Transformations.map(seekingRuleType) {
            if (Flavor().shouldStopGoToLiveInPlayer)
                View.VISIBLE
            else
                View.GONE
        }
        buttonOpenSubtitlesVisible = Transformations.map(seekingRuleType) {
            if (Flavor().shouldShowSubtitlesInfoInPlayer)
                View.VISIBLE
            else
                View.GONE
        }
        ageRatings = Transformations.map(innerDetail) {
            it.ageRatings
        }

        channelNumberAndTitle = Transformations.map(softZappChannel) {
            Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("EPG in player channel focused: ${it?.title} (CH:${it?.channelNumber})")
            getChannelNumberString(it)
        }

        channelIcon = Transformations.map(softZappChannel) {
            it ?: return@map EmptyImageSpecification()
            ExpirableGlideImageSpecification(
                    Flavor().pickBasedOnFlavor(Flavor().getImageOfType(
                            it,
                            ImageType.DARK
                    ),
                            channelLogoDesiredWidth.toInt(),
                            channelLogoDesiredHeight.toInt(),
                            SmartImages.UNRESTRICTED
                    ) ?: ""
            )
        }

        playerClockVisibility = Transformations.map(currentTimeMillis) {
            if (TimeProvider.isOutOfSync() || it == -1L) View.GONE else View.VISIBLE
        }

        when (Flavor().startOverIconDisplayRule) {
            StartOverIconDisplayRule.NEVER -> {
                playingEventStartOverVisible = MutableLiveData(View.GONE)
                nextEventStartOverVisible = MutableLiveData(View.GONE)
            }
            StartOverIconDisplayRule.ANYTIME_BASED_ON_SEEKING_RULE -> {
                playingEventStartOverVisible = Transformations.map(currentEvent) {
                    getVisibilityBasedOnSeekingRuleType(it?.seekingRuleType)
                }
                nextEventStartOverVisible = Transformations.map(nextEvent) {
                    getVisibilityBasedOnSeekingRuleType(it?.seekingRuleType)
                }
            }
            StartOverIconDisplayRule.ONLY_PAST_BASED_ON_SEEKING_RULE -> {
                playingEventStartOverVisible = Transformations.map(currentEvent) {
                    getPastVisibilityBasedOnSeekingRuleType(it?.seekingRuleType, it?.startDate)
                }
                nextEventStartOverVisible = Transformations.map(nextEvent) {
                    getPastVisibilityBasedOnSeekingRuleType(it?.seekingRuleType, it?.startDate)
                }
            }
            StartOverIconDisplayRule.CATCHUP_BASED -> {
                playingEventStartOverVisible = Transformations.map(currentEvent) {
                    getVisibilityBasedOnCatchup(it)
                }
                nextEventStartOverVisible = Transformations.map(nextEvent) {
                    getVisibilityBasedOnCatchup(it)
                }
            }
        }

        playingEventTitle = Transformations.map(currentEvent) { current ->
            val title = current?.title ?: Flavor().playerNoDataString
            if(current !is DummySmartMediaItem && current != null) Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("EPG in player event focused: $title (T:${current?.startDate?.millis}-${current?.endDate?.millis}) (CH:${softZappChannel.value?.channelNumber})")
           title
        }

        nextPlayingEventTitle = Transformations.map(nextEvent) { next ->
            next?.title ?: Flavor().playerNoDataString
        }
        playingEventTime = Transformations.map(currentEvent) {
            it ?: return@map ""
            dateTimeRepository.formatDateRange(it.startDate, it.endDate)
        }
        nextPlayingEventTime = Transformations.map(nextEvent) {
            it ?: return@map ""
            dateTimeRepository.formatDateRange(it.startDate, it.endDate)
        }

        playingEventRecording = CombinationTransformations.combineNullable(recordingState, forceUpdateLD) { state, _ ->
            when (state) {
                MetadataViewModel.RecordingState.WILL_BE_RECORDED,
                MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED -> View.VISIBLE
                else -> View.GONE
            }
        }

        nextPlayingEventRecording = CombinationTransformations.combineNullable(nextEvent, forceUpdateLD) { next, _ ->
            if (next == null || next is DummySmartMediaItem || next.reference is DummySmartMediaReference)
                View.GONE
            else {
                next.let { if (isNextEventInRecording(it)) View.VISIBLE else View.GONE }
            }
        }

        forceUpdateLD.value = forceUpdateLD.value?.inc()
    }

    fun setEpgFocusedEventStore(epgFocusedEventStore: EpgView.EpgFocusedEventStore?) {
        this.epgFocusedEventStore = epgFocusedEventStore
    }

    private fun storeEventAndChannelPosition(event: SmartMediaItem?, channelPosition: Int?) {
        if (event == null || channelPosition == null) {
            epgFocusedEventStore?.storeEventAndChannelPosition(null, 0)
        } else {
            val eventFromSmartMediaItem = EpgEvent(
                    null,
                    event.startDate?.millis ?: 0,
                    event.endDate?.millis ?: 0,
                    event.title
            )
            epgFocusedEventStore?.storeEventAndChannelPosition(
                    eventFromSmartMediaItem,
                    channelPosition
            )
        }
    }

    fun getPlayingEventTitleSafe(): String {
        return playingEvent.value?.title ?: ""
    }

    //region PlayerUIViewModel overrides

    @Suppress("SpellCheckingInspection")
    override fun seekToSelectedPos(player: SmartPlayer?, isInTrick: Boolean) {
        if (isPlayingCatchup) {
            barProgress.value?.also { player?.seekTo(it.toLong()) }
        } else {
            val dur = player?.duration
            val secprog = barSecondaryProgress.value
            val priprog = barProgress.value
            if (dur != null && secprog != null && priprog != null) {
                player.seekTo(dur - secprog + priprog)
            }
            if(!isInTrick) seeking = false
        }
    }

    override fun getSeekingRuleTypeInternal() = seekingRuleType.value

    override fun resetPauseTime() {
        pauseTimeMillis = 0
    }

    override fun setPauseTime() {
        pauseTimeMillis = TimeProvider.nowMs() - barStreamBehindLive
    }

    override fun onPlayerTimeUpdate(position: Long, duration: Long) {
        if (isPlayingCatchup) {
            //in case of softzapp just return because progress will be updated by other mechanism
            if (!softZaping()) {
                super.onPlayerTimeUpdate(position, duration)
            }
            return
        }
        barStreamBehindLive = if (duration > 0) duration - position else 0
    }

    override fun onPlayerActionStop() {
        updateTime = false
    }

    override fun startSeekingPrepare() {
        cancelSoftZap()
        seeking = true
    }

    override fun seekTick(streamDuration: Long?, seekIncrement: Int, seekMultiplier: Int): Int {
        var newSeekMultiplier = seekMultiplier
        barProgress.value?.let { primaryProgress ->
            var currentPos = primaryProgress
            currentPos += seekIncrement
            if (currentPos <= 0) {
                currentPos = 0
                if (seekMultiplier < 0) newSeekMultiplier = 0
            }

            if (isPlayingCatchup) {
                streamDuration?.let {
                    if (currentPos > it) {
                        currentPos = it.toInt()
                        if (seekMultiplier > 0) newSeekMultiplier = 0
                    }
                }
            } else {
                barSecondaryProgress.value?.let { livePos ->
                    if (currentPos >= livePos) {
                        currentPos = livePos
                        if (seekMultiplier > 0) newSeekMultiplier = 0
                    }
                    streamDuration?.let {
                        if (currentPos <= livePos - it) {
                            currentPos = (livePos - it).toInt()
                            if (seekMultiplier < 0) newSeekMultiplier = 0
                        }
                    }
                }
                barMax.value?.let {
                    if (currentPos >= it) {
                        currentPos = it
                        if (seekMultiplier > 0) newSeekMultiplier = 0
                    }
                }
            }
            playerPosition.value = (currentPos.toLong())
        }
        return newSeekMultiplier
    }

    override fun getNextChannelReference() = channels.value?.getOrNull(getNextChannelIndex(false))?.reference

    override fun getNextChannelIdTo(channelReference: SmartMediaReference): String? {
        channels.value?.forEachIndexed { index, channel ->
            if (channel.reference == channelReference) {
                val nextChannelIndex = index + 1
                return if (nextChannelIndex == channels.value?.size) {
                    "-1"
                } else {
                    channels.value?.get(nextChannelIndex)?.reference.toString()
                }
            }
        }
        return super.getNextChannelIdTo(channelReference)
    }

    override fun getPreviousChannelReference() = channels.value?.getOrNull(getPreviousChannelIndex(false))?.reference

    override fun getPreviousChannelIdTo(channelReference: SmartMediaReference): String? {
        channels.value?.forEachIndexed { index, channel ->
            if (channel.reference == channelReference) {
                val previousChannelIndex = index - 1
                return if (index == 0) {
                    "-1"
                } else {
                    channels.value?.get(previousChannelIndex)?.reference.toString()
                }
            }
        }
        return super.getPreviousChannelIdTo(channelReference)
    }

    override fun softZapNextChannel() {
        softZappChannel.value = channels.value?.getOrNull(getNextChannelIndex(true))
        softZappDirection.value = 0
    }

    override fun softZapPreviousChannel() {
        softZappChannel.value = channels.value?.getOrNull(getPreviousChannelIndex(true))
        softZappDirection.value = 0
    }

    override fun softZapNextEvent() {
        softZappDirection.value = 1
    }

    override fun softZapPreviousEvent() {
        softZappDirection.value = -1
    }

    override fun getSoftZapMediaReference(): SmartMediaReference? {
        softZaping().ifFalse { return null }

        softZappChannel.value?.let {
            if (Flavor().getAppChannelsDelegate()?.getAppChannelIDs()?.contains(it.reference.toString()) == true) {
                return it.reference
            }
        }

        currentEvent.value?.let {
            if (it.startDate == null || it.endDate == null)
                return softZappChannel.value?.reference
            val now = TimeProvider.now()
            if (now > it.startDate && now < it.endDate)
                return softZappChannel.value?.reference
            if (now > it.endDate) {
                return when {
                    Flavor().startOverIconDisplayRule != StartOverIconDisplayRule.CATCHUP_BASED -> currentEvent.value?.reference
                    it.isCatchup().not() -> null
                    else -> currentEvent.value?.reference
                }
            }
        } ?: return softZappChannel.value?.reference
        return null
    }

    override fun cancelSoftZap() {
        if (softZaping()) {
            if (softZappChannel.value != playingChannel.value) softZappChannel.value = playingChannel.value
            softZappDirection.value = 0
        }
    }

    override fun softZaping(): Boolean {
        return softZappDirection.value != 0 || softZappChannel.value != playingChannel.value
    }

    override fun getChannelReferenceByNumber(number: String): SmartMediaReference? {
        if (playingChannel.value?.channelNumber == number.toInt()) return null
        return getChannelByNumber(number.toInt())?.reference
    }

    override fun getClosestChannelReferenceByNumber(number: String): SmartMediaReference? {
        if (playingChannel.value?.channelNumber == number.toInt()) return null
        return getClosestChannelByNumber(number.toInt(), channels.value)?.reference
    }

    override fun getLastChannelReference(fromCatchup: Boolean): SmartMediaReference? {
        if (fromCatchup) return playingChannel.value?.reference
        if (lastPlayedChannelNumber == playingChannel.value?.channelNumber) return null
        return lastPlayedChannelNumber?.let { getChannelByNumber(it)?.reference }
    }

    override fun cancelSeeking() {
        seeking = false
    }

    override fun getPlayingEventStart(): Long {
        playingEvent.value?.startDate?.let {
            return it.millis - EVENT_SIZE_ADDITION
        } ?: run {
            return TimeProvider.nowMs()
        }
    }

    override fun isTimeShifted(): Boolean {
        return barStreamBehindLive > STREAM_BEHIND_LIVE_TRESHOLD
    }

    override fun observeMediaStreamLD(
            detail: SmartMediaDetail,
            channel: SmartMediaItem?,
            doBlock: (stream: SmartMediaStream) -> Unit,
            catchBlock: suspend CoroutineScope.(e: Exception) -> Boolean,
            lifecycleOwner: LifecycleOwner,
            isPlayingCatchup: Boolean
    ) {
        this.isPlayingCatchup = isPlayingCatchup
        isPlayingCatchup.ifTrue {
            playingEvent.replaceAllSource(epgRepository.getDetailLD(currentEvent.value, "playingEvent"), Observer{
                playingEvent.value = it
                //we need to set played channel, because the catchup on one channel might have been triggered via softzapping from entirely different channel
                playingChannel.value = channel
            })
            super.observeMediaStreamLD(detail, channel, doBlock, catchBlock, lifecycleOwner, isPlayingCatchup)
            return
        }
        playFailedVisibility.postValue(View.GONE)
        playFailedText.postValue(resourceRepository.getString(R.string.play_failed_channel)) //reset default error text
        channel.ifNull {
            errorState = true
            playFailedVisibility.postValue(View.VISIBLE)
            playFailedText.postValue(resourceRepository.getString(R.string.play_failed_subscription_toast))
            return
        }
        setPlayingChannel(channel!!)
        val newCatchBlock: suspend CoroutineScope.(e: Exception) -> Boolean =
            newCatchBlock@{ e: Exception ->
                GlobalScope.launch(Dispatchers.IO) { youboraAnalytics.reportPlayerError(e) }
                if (e is PlaybackRestrictedException) {
                    errorState = true
                    RestrictionChecker.handlePlaybackRestrictions(
                        exception = e,
                        playingChannel = channel,
                        isPlayingCatchup = isPlayingCatchup,
                        playFailedVisibility = playFailedVisibility,
                        playFailedText = playFailedText,
                        isCatchupSubscribed = isCatchupSubscribed,
                        subscriptionChannel = subscriptionChannel,
                        resourceRepository = resourceRepository
                    )
                    return@newCatchBlock true
                }
                return@newCatchBlock catchBlock(this, e)
            }
        delayedChannelStartTimer?.cancel()
        delayedChannelStartTimer = null
        delayedChannelStartTimer = Timer(CHANNEL_DELAY_TIMER, false).schedule(CHANNEL_DELAY_LENGTH) {
            CoroutineScope(Dispatchers.Main).launch {
                val mediaStreamLD = metadataRepository.getMediaStreamLD(
                        channel.reference,
                        newCatchBlock
                )
                mediaStreamLD.observe(lifecycleOwner, Observer { stream ->
                    doBlock.invoke(stream)
                    mediaStreamLD.removeObservers(lifecycleOwner)
                })
            }
        }
    }

    override fun setPlayingChannel(channel: SmartMediaItem) {
        barStreamBehindLive = 0
        pauseTimeMillis = 0
        updateTime = true
        errorState = false
        futureNextEvent = null
        shouldCacheFutureNextEvent = true
        channel.channelNumber.also { number ->
            if (number != playingChannel.value?.channelNumber) lastPlayedChannelNumber = playingChannel.value?.channelNumber
        }
        playingChannel.value = channel
        softZappChannel.value = channel
        playingEvent.replaceAllSource(epgRepository.getNowEventAndDetailFromChannelLDs(playingChannel.value).second, Observer {
            playingEvent.value = it ?: epgRepository.noDataSmartMediaItem
            softZappDirection.value = 0
        })
    }

    override fun getPlayingEventTitleLD(): LiveData<String> = playingEventTitle

    //endregion

    fun setObserversInLifecycle(lifecycleOwner: LifecycleOwner) {
        currentTimeMillis.observe(lifecycleOwner, Observer {
            if (updateTime) updateTime()
        })

        currentEvent.observe(lifecycleOwner, Observer {
            updateProgressBar(currentEvent.value)
        })

        playingEvent.observe(lifecycleOwner, Observer {
            if (playingEvent.value != null && playingEvent.value?.restrictions != null) {
                detail.value?.restrictions = playingEvent.value?.restrictions!!.toMutableList()
            }
            updateProgressBar(playingEvent.value, EVENT_SIZE_ADDITION)
            storeEventAndChannelPosition(playingEvent.value, getChannelIndex(false))
        })

        softZappDirection.observe(lifecycleOwner, Observer {
            when (it) {
                -1 -> {
                    if(currentEvent.value == null || currentEvent.value is DummySmartMediaItem) return@Observer
                    nextEvent.removeAllSourcesAndSetValue(currentEvent.value, true)
                    currentEvent.replaceAllSourceAndMap(epgRepository.getPrevEventLD(currentEvent.value, "currentEvent"))
                }
                1 -> {
                    if(nextEvent.value == null || nextEvent.value is DummySmartMediaItem) return@Observer
                    currentEvent.removeAllSourcesAndSetValue(nextEvent.value, true)
                    nextEvent.replaceAllSourceAndMap(epgRepository.getNextEventLD(nextEvent.value, "nextEvent"))
                }
                else -> {
                    if(softZaping()) {
                        currentEvent.replaceAllSourceAndMap(epgRepository.getNowEventAndDetailFromChannelLDs(softZappChannel.value).first) { current ->
                            nextEvent.replaceAllSourceAndMap(epgRepository.getNextEventLD(current, "nextEvent"))
                        }
                    } else {
                        currentEvent.replaceAllSourceAndMap(playingEvent) { current ->
                            val innerNextEvent = futureNextEvent ?: epgRepository.getNextEventLD(current, "nextEvent")
                            nextEvent.replaceAllSourceAndMap(innerNextEvent)
                            shouldCacheFutureNextEvent = true
                            futureNextEvent = null
                        }
                    }
                }
            }
        })
    }

    private fun updateTime() {
        isPlayingCatchup.ifTrue { return }
        val currentTime = TimeProvider.nowMs()
        softZaping().ifFalse {
            val startMillis = playingEvent.value?.startDate?.millis ?: 0
            val endMillis = playingEvent.value?.endDate?.millis ?: 0
            playerBufferedPosition.value = (currentTime - startMillis + EVENT_SIZE_ADDITION)
            if (pauseTimeMillis > 0) {
                if (!seeking) playerPosition.value = (pauseTimeMillis - startMillis + EVENT_SIZE_ADDITION)
            } else {
                if (shouldGetFutureNextEvent(startMillis, endMillis, currentTime)) {
                    futureNextEvent = epgRepository.getNextEventLD(nextEvent.value, "nextEvent")
                    shouldCacheFutureNextEvent = false
                }
                if (endMillis > 0 && startMillis > 0 && currentTime - barStreamBehindLive > endMillis) {
                    //TODO fix if it is problem
                    //this will force reload events so jump to next event, known issue is that if stream is paused longer then one event, this will jump to currently airing event and not to next one
                    playingEvent.replaceAllSource(epgRepository.getNowEventAndDetailFromChannelLDs(playingChannel.value).second, Observer {
                        playingEvent.value = it
                        softZappDirection.value = 0
                    })
                } else {
                    playerPosition.value = (currentTime - startMillis + EVENT_SIZE_ADDITION - barStreamBehindLive)
                }
            }
        }
    }

    private fun shouldGetFutureNextEvent(startMillis: Long, endMillis: Long, currentTime: Long): Boolean =
        shouldCacheFutureNextEvent && endMillis > 0 && startMillis > 0 && currentTime - barStreamBehindLive > (endMillis - (randomMinuteToCacheFutureNextEvent * 60000))

    private fun getVisibilityBasedOnSeekingRuleType(seekingRuleType: SmartSeekingRuleType?): Int {
        return when (seekingRuleType) {
            null,
            SmartSeekingRuleType.LIVE_ONLY -> View.GONE
            SmartSeekingRuleType.STARTOVER,
            SmartSeekingRuleType.TIMESHIFT,
            SmartSeekingRuleType.LIVE_STARTOVER,
            SmartSeekingRuleType.LIVE_TIMESHIFT,
            SmartSeekingRuleType.CATCHUP_STARTOVER,
            SmartSeekingRuleType.CATCHUP_TIMESHIFT -> View.VISIBLE
        }
    }

    private fun getPastVisibilityBasedOnSeekingRuleType(seekingRuleType: SmartSeekingRuleType?, startDate: DateTime?): Int {
        return if (startDate == null || startDate.millis > TimeProvider.nowMs()) {
            View.GONE
        } else {
            getVisibilityBasedOnSeekingRuleType(seekingRuleType)
        }
    }

    private fun getVisibilityBasedOnCatchup(event: SmartMediaItem?): Int {
        if (event?.endDate?.millis ?: 0 < TimeProvider.nowMs()) {
//            Log.d("FlowApi", "getVisibilityBasedOnCatchup ${event?.isCatchup()}")
        }
        if(event?.isCatchup() == true) return View.VISIBLE
        return View.GONE
    }

    private fun getChannelNumberString(channel: SmartMediaItem?): String {
        channel ?: return ""
        return Flavor().getPlayerControlsChannelTitle(channel.channelNumber.toString(), channel.title ?: "")
    }

    private fun updateProgressBar(event: SmartMediaItem?, eventSizeAddition: Int = 0) {
        if (isPlayingCatchup && !softZaping()) {
            //set clean values because real ones will be set by periodic update from player
            //this ones are used just for clean during soft zapp when catchup is playing
            playerDuration.value = (100)
            playerBufferedPosition.value = (0)
            playerPosition.value = (0)
            return
        }
        event?.also {
            val currentTime = TimeProvider.nowMs()
            val startMillis = event.startDate?.millis ?: 0
            val endMillis = event.endDate?.millis ?: 0
            playerDuration.value = (endMillis - startMillis + eventSizeAddition)
            playerBufferedPosition.value = when (isPlayingCatchup) {
                true -> 0
                false -> (currentTime - startMillis + eventSizeAddition)
            }
            playerPosition.value = when (isPlayingCatchup) {
                true -> 0
                false -> (currentTime - startMillis + eventSizeAddition)
            }
            return
        }
        playerDuration.value = (-1)
    }

    private fun isCurrentEventInRecording(event: SmartMediaItem): Boolean {
        val recording = recordings.value?.find { Flavor().compareRecordingWithEvent(it, event) }
        return if (recording != null) {
            innerDetail.value?.recordingReference = recording.reference
            true
        } else
            false
    }

    private fun isNextEventInRecording(event: SmartMediaItem): Boolean {
        return recordings.value?.find { Flavor().compareRecordingWithEvent(it, event) } != null
    }

    private fun getChannelIndex(channels: List<SmartMediaItem>?, channel: SmartMediaItem?): Int? {
        val index = channels?.indexOf(channel)
        if (index != null && index >= 0) return index
        return null
    }

    private fun getChannelIndex(softZap: Boolean): Int? {
        return getChannelIndex(channels.value, if (softZap) softZappChannel.value else playingChannel.value)
    }

    private fun getNextChannelIndex(softZap: Boolean): Int {
        getChannelIndex(softZap)?.let { playingIndex ->
            channels.value?.let {
                return if (playingIndex >= it.size - 1) 0
                else playingIndex + 1
            }
        }
        return 0
    }

    private fun getPreviousChannelIndex(softZap: Boolean): Int {
        getChannelIndex(softZap)?.let { playingIndex ->
            channels.value?.let {
                return if (playingIndex == 0) it.size - 1
                else playingIndex - 1
            }
        }
        return 0
    }

    fun startStopRecording(lifecycleOwner: LifecycleOwner) {
        innerDetail.value?.channelReference = playingEvent.value?.channelReference
        innerDetail.value?.programReference = playingEvent.value?.programReference
        when (if (useOriginalRecordingState) recordingState.value else userTriggeredRecordingState.value) {
            MetadataViewModel.RecordingState.CAN_BE_AUTO_RECORDED -> RecordingsDialogsHelper.showPlanSeriesRecordingDialog(
                resourceRepository,
                { startRecording(lifecycleOwner, false) },
                { startRecording(lifecycleOwner, true) }
            )
            MetadataViewModel.RecordingState.CAN_BE_RECORDED -> startRecording(lifecycleOwner, false)
            MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED -> RecordingsDialogsHelper.showUnplanSeriesRecordingDialog(
                resourceRepository,
                { stopRecording(lifecycleOwner, false) },
                { stopRecording(lifecycleOwner, true) }
            )
            MetadataViewModel.RecordingState.WILL_BE_RECORDED -> stopRecording(lifecycleOwner, false)
            MetadataViewModel.RecordingState.WAS_AUTO_RECORDED -> RecordingsDialogsHelper.showDeleteSeriesRecordingDialog(
                resourceRepository,
                { deleteRecording(lifecycleOwner, false) },
                { deleteRecording(lifecycleOwner, true) }
            )
            MetadataViewModel.RecordingState.WAS_RECORDED -> deleteRecording(lifecycleOwner, false)
            else -> {
                useOriginalRecordingState = false
                userTriggeredRecordingState.postValue(recordingState.value)
            }
        }
    }

    private fun startRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {

        val storageFullCatchBlock = recordingsRepository.getStorageFullCatchBlock(resourceRepository)
        recordingsRepository.startRecordingLD(innerDetail.value, autoRecording, catchBlock = storageFullCatchBlock).observe(
            lifecycleOwner,
            Observer {
                useOriginalRecordingState = false
                userTriggeredRecordingState.postValue(MetadataViewModel.RecordingState.WILL_BE_RECORDED)
                innerDetail.value?.recordingReference = if (autoRecording) it.recordingReference else it.reference
                playingEventRecording.postValue(View.VISIBLE)
                //recordings must be refreshed after live recording change
                recordingsRepository.refreshRecordings()
            }
        )
    }

    private fun stopRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {
        recordingsRepository.stopRecordingLD(innerDetail.value,
            autoRecording = false,
            removeRecorded = false
        ).observe(
            lifecycleOwner,
            Observer {
                playingEventRecording.postValue(View.GONE)
                recordingsRepository.refreshRecordings()
                useOriginalRecordingState = false
                userTriggeredRecordingState.postValue(
                    if (autoRecording) MetadataViewModel.RecordingState.CAN_BE_AUTO_RECORDED
                    else MetadataViewModel.RecordingState.CAN_BE_RECORDED
                )
            }
        )
    }

    private fun deleteRecording(lifecycleOwner: LifecycleOwner, autoRecording: Boolean) {
        recordingsRepository.stopRecordingLD(innerDetail.value,
            autoRecording = false,
            removeRecorded = true
        ).observe(
            lifecycleOwner,
            Observer {
                playingEventRecording.postValue(View.GONE)
                recordingsRepository.refreshRecordings()
                useOriginalRecordingState = false
                userTriggeredRecordingState.postValue(MetadataViewModel.RecordingState.NOTHING)
            }
        )
    }

    private fun SmartMediaItem.getRecordingState(recordings: List<SmartMediaItem>): MetadataViewModel.RecordingState {
        val myRecordingObj = recordings.find { belongsToRecording(it) } //this checks only the non-auto recordings
        val isInFuture = isInFuture()
        val myAutoRecordingObj = myRecordingObj?.recordingReference ?: //check for auto-recording
        recordings.find { belongsToAutoRecording(it) }
        val isInPast = endDate?.millis ?: 0 < TimeProvider.now().millis
        return when {
            myRecordingObj != null && isInFuture && myAutoRecordingObj != null -> MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED
            myRecordingObj != null && isInFuture -> MetadataViewModel.RecordingState.WILL_BE_RECORDED
            myRecordingObj != null && isInPast && myAutoRecordingObj != null -> MetadataViewModel.RecordingState.WAS_AUTO_RECORDED
            myRecordingObj != null && isInPast -> MetadataViewModel.RecordingState.WAS_RECORDED
            myRecordingObj == null && isInFuture && seriesReference != null && myAutoRecordingObj == null -> MetadataViewModel.RecordingState.CAN_BE_AUTO_RECORDED
            myRecordingObj == null && isInFuture -> MetadataViewModel.RecordingState.CAN_BE_RECORDED
            else -> MetadataViewModel.RecordingState.NOTHING
        }
    }

    private fun SmartMediaItem.isRecordingAllowed(): Boolean
            = restrictions.find { it.type == SmartRestrictionType.RECORDING } == null

    private fun SmartMediaItem.isInFuture(): Boolean {
        return if (Flavor().isLiveRecordingAllowed)
            (endDate?.millis ?: 0 > TimeProvider.now().millis)
        else
            (startDate?.millis ?: 0 > TimeProvider.now().millis)
    }

    private fun SmartMediaItem.belongsToRecording(recording: SmartMediaItem): Boolean {
        return when {
            type == SmartMediaType.RECORDING -> recording.reference == reference
            recording.type == SmartMediaType.AUTO_RECORDING -> false
            else -> Flavor().compareRecordingWithEvent(recording, this)
        }
    }

    private fun SmartMediaItem.belongsToAutoRecording(recording: SmartMediaItem): Boolean {
        if (recording.type != SmartMediaType.AUTO_RECORDING) return false
        if (seriesReference == null) return false
        return seriesReference == recording.seriesReference
    }

    private fun SmartMediaItem.isCatchup()
            = endDate?.millis != null
            && endDate!!.millis < TimeProvider.nowMs() //end time accuracy to minutes, needed for catchup flag
            && seekingRuleAllowsCatchup()
            && Flavor().isCatchupStreamAvailable(this)

    override fun isSoftZappedPlayableCatchup() = currentEvent.value?.isCatchup() == true
}