package com.twentyfouri.tvlauncher.viewmodels

import android.view.View
import androidx.lifecycle.*
import androidx.lifecycle.Transformations.map
import com.twentyfouri.apicommon.HTTPStatus
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.error.AppCanRunWithLimitationsException
import com.twentyfouri.smartmodel.model.error.AppCannotRunException
import com.twentyfouri.smartmodel.model.error.InvalidCredentialsException
import com.twentyfouri.smartmodel.model.error.WrongInputException
import com.twentyfouri.smartmodel.model.error.*
import com.twentyfouri.smartmodel.model.menu.SmartMenu
import com.twentyfouri.tvlauncher.BuildConfig
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.PageType
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.common.data.SmartApiRepository
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogAction
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragmentListener
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.MenuRepository
import com.twentyfouri.tvlauncher.ui.OnCreateState
import com.twentyfouri.tvlauncher.utils.Navigator
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import timber.log.Timber
import java.io.File
import java.io.IOException

class MainActivityViewModel(
    private val setupDataRepository: SetupDataRepository,
    private val menuRepository: MenuRepository,
    private val epgRepository: EpgRepository,
    private val smartModelApi: FlowSmartApi?,
    val apiHandler: ApiHandler,
    private val youboraAnalytics: YouboraAnalytics
) : ViewModel() {

    private val onCreateStateInternal = MutableLiveData<OnCreateState>()
    private val _recordingClicked = MutableLiveData<Unit>()
    private val _recordingInPlayerClicked = MutableLiveData<Unit>()
    private val isLoading = MutableLiveData<Boolean>().apply { value = false }
    private val showSplash = MutableLiveData<Boolean>().apply { value = true }
    private val _showDebugButtons = MutableLiveData<Boolean>().apply { value = true }

    private val onCreateStateExternal: LiveData<OnCreateState>
    private val menus: LiveData<List<SmartMenu>>
    val recordingClicked: LiveData<Unit> = _recordingClicked
    val recordingInPlayerClicked: LiveData<Unit> = _recordingInPlayerClicked
    val logoutText = getUserNameLD()
    val progressLayoutVisibility: LiveData<Int>
    val contentFrameVisibility: LiveData<Int>
    val splashScreenVisibility: LiveData<Int>
    val showDebugButtons: LiveData<Boolean> = _showDebugButtons
    val debugButtonsVisibility: LiveData<Int>
    private var apiExceptionMessage: Exception? = null

    init {
        menus = Transformations.switchMap(onCreateStateInternal) { state ->
            val offline = when (state) {
                OnCreateState.NOT_ALLOWED,
                OnCreateState.LOGGED_IN,
                OnCreateState.LOGGED_IN_WITH_LIMITATIONS,
                OnCreateState.RESTART -> false
                OnCreateState.OFFLINE_AND_SETUP_WIZARD,
                OnCreateState.OFFLINE_AND_NETWORK_ERROR,
                OnCreateState.OFFLINE_AND_LOGIN_ERROR,
                OnCreateState.OFFLINE_AND_TOO_MANY_REQUESTS_ERROR/*,
                OnCreateState.UNINITIALIZED*/ -> true
                null -> true
            }
            menuRepository.getMenusLD(offline)
        }
        onCreateStateExternal = Transformations.switchMap(menus) { //wait on menu
            //TODO base fragment triggering should be enough
            //if(onCreateStateInternal.value == OnCreateState.LOGGED_IN) epgRepository.enqueuePopulateDatabase()
            onCreateStateInternal
        }
        debugButtonsVisibility = map(_showDebugButtons) {
            if (BuildConfig.BUILD_TYPE == "debug" && it) {
                View.VISIBLE
            } else {
                View.GONE
            }
        }
        progressLayoutVisibility = map(isLoading) { if (it) View.VISIBLE else View.GONE }
        contentFrameVisibility = map(isLoading) { if (it) View.INVISIBLE else View.VISIBLE }
        splashScreenVisibility = map(showSplash) { if (it) View.VISIBLE else View.GONE }
    }

    @Suppress("ConstantConditionIf", "unused")
    fun getLogoutVisibility() = if (BuildConfig.BUILD_TYPE == "debug") View.VISIBLE else View.GONE

    fun getLogoutOnClick() = View.OnClickListener {
        Timber.d("getLogoutOnClick()")
        onCreateStateInternal.value = OnCreateState.OFFLINE_AND_SETUP_WIZARD
    }

    fun getRestartOnClick() = View.OnClickListener {
        Timber.d("getRestartOnClick()")
        onCreateStateInternal.value = OnCreateState.RESTART
    }

    fun getDoSomethingOnClick() = View.OnClickListener {
        val messageDialogModel = MessageDialogModel(
                "message",
                "description",
                arrayOf("optionA", "optionB"),
                "cancel",
                "code"
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        dialogFragment.mListener = MessageDialogFragmentListener.from { messageDialogResult: MessageDialogAction -> true }
        dialogFragment.isCancelable = true
        Navigator.getInstance().navigateDialog(dialogFragment)
    }

    private fun getUserNameLD(): LiveData<String> {
        val name = MutableLiveData<String>()
        apiHandler.joinPreviousOrLaunchNew(
            synchronizedJobName = "getUserNameLD",
            block = {
                val setupData = setupDataRepository.getSetupData()
                val ss = com.twentyfouri.tvlauncher.common.Flavor().getUsername(setupData)
                name.postValue("Logout user $ss")
            }
        )
        return name
    }

    fun viewModelScope() = viewModelScope

//    fun resetAndGetOnCreateStateLD(): LiveData<OnCreateState> {
//        onCreateStateInternal.postValue(OnCreateState.UNINITIALIZED)
//        return getOnCreateStateLD()
//    }

    fun getOnCreateStateLD(): LiveData<OnCreateState> {
        apiHandler.joinPreviousOrLaunchNew(
            synchronizedJobName = "getOnCreateStateLD",
            block = {
                if (smartModelApi == null) {
                    onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_NETWORK_ERROR)
                } else {
                    Flavor().setupSmartApiAnalyticsListener(smartModelApi, youboraAnalytics)
                    onCreateStateInternal.postValue(
                        when (Flavor().isLoggedIn(smartModelApi)){
                            true -> {
                                SmartApiRepository.isServerBusy = false
                                Flavor().checkDeviceLocation(smartModelApi)
                                com.twentyfouri.tvlauncher.common.Flavor().overrideEditionMapping(smartModelApi)
                                CoroutineScope(Dispatchers.IO).launch {
                                    youboraAnalytics.setupUser(smartModelApi)
                                }
                                OnCreateState.LOGGED_IN
                            }
                            else -> {
                                SmartApiRepository.isServerBusy = false
                                OnCreateState.OFFLINE_AND_SETUP_WIZARD
                            }
                        }
                    )
                }
            },
            catchBlock = {
                when (it) {
                    is AppCannotRunException -> {
                        onCreateStateInternal.postValue(OnCreateState.NOT_ALLOWED)
                        false
                    }
                    is AppCanRunWithLimitationsException -> {
                        onCreateStateInternal.postValue(OnCreateState.LOGGED_IN_WITH_LIMITATIONS)
                        true
                    }
                    is WrongInputException, is InvalidCredentialsException -> {
                        if (Flavor().allowOfflineIfLoginFails()) {
                            onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_NETWORK_ERROR)
                            true
                        } else {
                            onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_LOGIN_ERROR)
                            false
                        }
                    }
                    is InvalidSessionException -> {
                        onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_LOGIN_ERROR)
                        false
                    }
                    else -> {
                        if (it is GeneralApiException && it.code == HTTPStatus.TooManyRequests.statusCode.toString()) {
                            SmartApiRepository.isServerBusy = true
                            onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_TOO_MANY_REQUESTS_ERROR)
                            true
                        } else {
                            apiExceptionMessage = it
                            onCreateStateInternal.postValue(OnCreateState.OFFLINE_AND_NETWORK_ERROR)
                            false
                        }
                    }
                }
            }
        )
        return onCreateStateExternal
    }

    fun getPageReference(pageType: PageType) = Flavor().getPageReference(pageType, menus.value)

    fun recordingClicked() { _recordingClicked.value = Unit }

    fun recordingInPlayerClicked() { _recordingInPlayerClicked.value = Unit }

    fun showProgress(show: Boolean) { isLoading.postValue(show && splashScreenVisibility.value == View.GONE) }

    fun showSplash(show: Boolean) { showSplash.postValue(show) }

    fun showDebugButtons(show: Boolean) { _showDebugButtons.postValue(show) }

    fun getApiExceptionMessage() = apiExceptionMessage
}