package com.twentyfouri.tvlauncher.viewmodels

import android.view.KeyEvent
import android.view.View
import androidx.lifecycle.*
import androidx.lifecycle.Transformations.map
import androidx.lifecycle.Transformations.switchMap
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.androidcore.utils.ImageSpecification
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.menu.SmartNavigationTarget
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.data.ResourceRepository
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.PlaylistType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.utils.RestrictionChecker
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.data.DateTimeRepository
import com.twentyfouri.tvlauncher.data.EpgRepository
import com.twentyfouri.tvlauncher.data.RowPageRepository
import com.twentyfouri.tvlauncher.extensions.isCatchup
import com.twentyfouri.tvlauncher.extensions.seekingRuleAllowsStartover
import com.twentyfouri.tvlauncher.ui.MainActivity
import com.twentyfouri.tvlauncher.ui.PlayerLocation
import com.twentyfouri.tvlauncher.utils.*
import org.joda.time.Seconds
import timber.log.Timber
import java.util.*

class DetailViewModel(
    private val navigator: Navigator,
    val dateTimeRepository: DateTimeRepository,
    epgRepository: EpgRepository?,
    val resourceRepository: ResourceRepository,
    val apiHandler: ApiHandler,
    val rowPageRepository: RowPageRepository
) : ViewModel() {
    companion object {
        const val ONE_SECOND: Long = 1000
    }

    private lateinit var mTimer: Timer

    private val mTimerValue = MutableLiveData<Int>()
    private val mediaDetail = MutableLiveData<SmartMediaDetail>()
    private val _mediaReference = MutableLiveData<SmartMediaReference>()
    private val _detailImageSpecification = MutableLiveData<ImageSpecification>()
    private val recordingState = MutableLiveData<MetadataViewModel.RecordingState>()
    private val channelReference = MutableLiveData<SmartMediaReference?>()
    private val _startStopRecordingClicked = MutableLiveData<Unit>()
    private val playability = MutableLiveData<List<MetadataViewModel.PlayabilityState>>()
    private val _shownMore = MutableLiveData<Boolean>().apply { value = false }
    private val _isExpanded = MutableLiveData<Boolean>().apply { value = false }
    private val _overallButtonsVisibility = MutableLiveData<Int>().apply { value = View.GONE }
    private val channelLogoDesiredWidth = resourceRepository.getPixelSizeFromDimens(R.dimen.detail_fragment_icon_width)
    private val channelLogoDesiredHeight = resourceRepository.getPixelSizeFromDimens(R.dimen.detail_fragment_icon_height)

    val mediaReference: LiveData<SmartMediaReference?> = _mediaReference
    val detailImageSpecification: LiveData<ImageSpecification> = _detailImageSpecification
    val playButtonProgressBarVisibility: LiveData<Int>
    val playButtonVisibility: LiveData<Int>
    val planRecordingButtonVisibility: LiveData<Int>
    val stopRecordingButtonVisibility: LiveData<Int>
    val deleteRecordingButtonVisibility: LiveData<Int>
    val channels: LiveData<List<SmartMediaItem>> = epgRepository?.getAllChannelsLD() ?: MutableLiveData()
    val pageSectionFromChannelReference: LiveData<SmartPageSection?>
    val channelImageSpecification: LiveData<ImageSpecification>
    val startStopRecordingClicked: LiveData<Unit> = _startStopRecordingClicked
    val restartButtonVisibility: LiveData<Int>
    val detailProgress: LiveData<Int>
    val rowViewVisibility: LiveData<Int>
    private val isLive: LiveData<Boolean?>
    val shownMore: LiveData<Boolean> = _shownMore
    val metadataTopMargin: LiveData<Int> = MutableLiveData<Int>().apply {
        value = resourceRepository.getDimensionPixelSize(R.dimen.detail_fragment_metadata_margin_top) }
    val isExpanded: LiveData<Boolean> = _isExpanded
    val showMoreButtonText: LiveData<String>
    val overallButtonsVisibility: LiveData<Int> = _overallButtonsVisibility
    val moreButtonFocusUp: LiveData<Int>
    val anyButtonVisibility: LiveData<Int>
    val sectionViewModelFromChannelReference: LiveData<SectionViewModel?>
    val rowViewModelsFromChannelReference: LiveData<List<RowViewModel>>
    val canBeStartedFromBeginning: LiveData<Boolean>

    val scrollUp = SingleLiveEvent<Void>()
    val scrollDown = SingleLiveEvent<Void>()
    val onGlobalLayout = SingleLiveEvent<Void>()

    init {
        playButtonProgressBarVisibility = map(mediaDetail) {
            if (it.endDate?.isAfterNow == true || it.positionInSeconds > 0) {
                setDetailButtonsVisibility(View.VISIBLE)
                View.VISIBLE
            } else View.GONE }
        playButtonVisibility = CombinationTransformations.combineNullable(recordingState, playability) { recordingState, playability ->
            when {
                recordingState == MetadataViewModel.RecordingState.WAS_RECORDED ||
                recordingState == MetadataViewModel.RecordingState.WAS_AUTO_RECORDED -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE }
                playability.isNullOrEmpty().not() -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE }
                else -> { View.GONE }
            }
        }
        isLive = map(playability) { it.contains(MetadataViewModel.PlayabilityState.IS_LIVE) || it.contains(MetadataViewModel.PlayabilityState.BROKEN_LIVE) }
        restartButtonVisibility = CombinationTransformations.combineNullable(_mediaReference, isLive, mediaDetail) { mediaReference, isLive, mediaDetail ->
            when {
                mediaReference == null -> View.GONE
                mediaDetail == null -> View.GONE
                isLive == true && mediaDetail.seekingRuleAllowsStartover(false) -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                mediaDetail.positionInSeconds <= 0 -> View.GONE
                mediaDetail.type != SmartMediaType.LIVE_EVENT -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                mediaDetail.isCatchup() -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                else -> View.GONE
            }
        }
        planRecordingButtonVisibility = map(recordingState) {
            when (it) {
                MetadataViewModel.RecordingState.CAN_BE_RECORDED,
                MetadataViewModel.RecordingState.CAN_BE_AUTO_RECORDED -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                else -> View.GONE
            }
        }
        stopRecordingButtonVisibility = map(recordingState) {
            when (it) {
                MetadataViewModel.RecordingState.WILL_BE_RECORDED,
                MetadataViewModel.RecordingState.WILL_BE_AUTO_RECORDED -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                else -> View.GONE
            }
        }
        deleteRecordingButtonVisibility = map(recordingState) {
            when (it) {
                MetadataViewModel.RecordingState.WAS_RECORDED,
                MetadataViewModel.RecordingState.WAS_AUTO_RECORDED -> {
                    setDetailButtonsVisibility(View.VISIBLE)
                    View.VISIBLE
                }
                else -> View.GONE
            }
        }
        channelImageSpecification = map(channelReference) { externalChannelReference ->
            channels.value?.find { it.reference == externalChannelReference }?.let {channel ->
                Flavor().pickBasedOnFlavor(Flavor().getImageOfType(channel, ImageType.DARK), channelLogoDesiredWidth.toInt(), channelLogoDesiredHeight.toInt(), SmartImages.UNRESTRICTED)?.let {
                    return@map ExpirableGlideImageSpecification(it)
                }
            }
            EmptyImageSpecification()
        }
        moreButtonFocusUp = CombinationTransformations.combineNullable(recordingState, playability) { recordingState, playability ->
            when {
                recordingState != MetadataViewModel.RecordingState.NOTHING ||
                        playability.isNullOrEmpty().not() -> R.id.detail_section_buttons
                else ->
                    R.id.show_more_button
            }
        }
        anyButtonVisibility = CombinationTransformations.map(
            {
                if (playButtonVisibility.value != View.VISIBLE
                    && planRecordingButtonVisibility.value != View.VISIBLE
                    && stopRecordingButtonVisibility.value != View.VISIBLE
                ) View.GONE
                else View.VISIBLE
            },
            listOf(
                playButtonVisibility,
                planRecordingButtonVisibility,
                stopRecordingButtonVisibility,
                recordingState,
                playability
            )
        )
        pageSectionFromChannelReference = map(channelReference) { reference ->
            reference ?: return@map null

            val channelName = channels.value?.find { it.reference == reference }?.title ?: ""
            if (com.twentyfouri.tvlauncher.common.Flavor().seriesRecordingsEnabled
                && (recordingState.value == MetadataViewModel.RecordingState.WAS_RECORDED
                        || recordingState.value == MetadataViewModel.RecordingState.WAS_AUTO_RECORDED)) {
                SmartPageSection(
                    "detail",
                    resourceRepository.getString(R.string.available_series_recordings),
                    Flavor().getPlaylistReference(PlaylistType.RECENT_RECORDINGS, mediaDetail.value?.seriesReference)
                ).apply {
                    sectionStyle = PlaylistType.RECENT_RECORDINGS.name }
            } else {
                SmartPageSection(
                    "detail",
                    resourceRepository.getString(R.string.detail_upcoming_label, channelName),
                    Flavor().getPlaylistReference(PlaylistType.UPCOMING_PROGRAMS, reference)
                ).apply { sectionStyle = PlaylistType.UPCOMING_PROGRAMS.name }
            }
        }

        detailProgress = CombinationTransformations.combineNullable(mediaDetail, mTimerValue, isLive) { detail, timerValue, isLive ->
            if (isLive == false) {
                return@combineNullable if (detail != null && detail.positionInSeconds > 0) {
                    Flavor().getBookmarkProgress(detail.positionInSeconds, detail.startDate, detail.endDate, detail.durationInSeconds)
                } else 0
            } else if (::mTimer.isInitialized.not()) {
                initializeTimerFromMediaDetail(detail)
                return@combineNullable 0
            }
            timerValue ?: 0
        }
        rowViewVisibility = CombinationTransformations.combineNullable(playability, _shownMore, recordingState) { playability, shownMore, recordingState ->
            if (playability?.contains(MetadataViewModel.PlayabilityState.IS_LIVE) == true && shownMore == false)
                View.VISIBLE
            else {
                if (com.twentyfouri.tvlauncher.common.Flavor().seriesRecordingsEnabled && (recordingState == MetadataViewModel.RecordingState.WAS_RECORDED || recordingState == MetadataViewModel.RecordingState.WAS_AUTO_RECORDED)) {
                    View.VISIBLE
                } else {
                    View.GONE
                }
            }
        }
        showMoreButtonText = map(_shownMore) { resourceRepository.getString(if (it) R.string.show_less_button else R.string.show_more_button) }
        sectionViewModelFromChannelReference = map(pageSectionFromChannelReference) {
            it ?: return@map null
            SectionViewModel(rowPageRepository, resourceRepository, epgRepository, it, 1)
        }
        rowViewModelsFromChannelReference = switchMap(sectionViewModelFromChannelReference) {
            it ?: return@switchMap MutableLiveData<List<RowViewModel>>()
            it.rowViewModels
        }
        canBeStartedFromBeginning = map(mediaDetail) {
            it.seekingRuleAllowsStartover(
                    isCatchup = null
            )
        }
    }

    private fun initializeTimerFromMediaDetail(detail: SmartMediaDetail?) {
        detail?.startDate?.let { start ->
            detail.endDate?.let { end ->
                mTimer = Timer()
                mTimer.scheduleAtFixedRate(
                    object : TimerTask() {
                        override fun run() {
                            val newValue = (100 * Seconds.secondsBetween(start, TimeProvider.now()).seconds) / Seconds.secondsBetween(start, end).seconds
                            if (newValue != detailProgress.value) mTimerValue.postValue(newValue)
                        }
                    },
                    ONE_SECOND,
                    ONE_SECOND
                )
            }
        }
    }

    fun setMediaReference(reference: SmartMediaReference) { _mediaReference.value = reference }

    fun setDetailImageSpecification(image: ImageSpecification) { _detailImageSpecification.value = image }

    fun setRecordingState(state: MetadataViewModel.RecordingState?) { recordingState.value = state ?: MetadataViewModel.RecordingState.NOTHING }

    fun setChannelReference(reference: SmartMediaReference?) { channelReference.value = reference }

    fun setMediaDetail(detail: SmartMediaDetail) { mediaDetail.value = detail }

    fun setPlayability(list: List<MetadataViewModel.PlayabilityState>) { playability.value = list }

    fun setIsExpanded(boolean: Boolean) { _isExpanded.value = boolean }

    fun setShownMore(boolean: Boolean) { _shownMore.value = boolean }

    fun setDetailButtonsVisibility(visibility: Int) {
        //block visibility set when description is expanded
        //once description is collapsed visibility is set again
        if(shownMore.value == true && visibility == View.VISIBLE) return
        _overallButtonsVisibility.value = visibility
    }

    override fun onCleared() {
        super.onCleared()
        if (::mTimer.isInitialized) mTimer.cancel()
        Timber.tag("leaks").d("DetailViewModel onCleared")
    }

    fun onPlayButtonClicked(v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) play button selected")
        ifCheckIsAdultContentLockedInvokeBlock(v) {
            val reference = if (isLive.value == true) mediaDetail.value?.channelReference else _mediaReference.value
            reference?.let {
                //from detail we probably always want foreground player
                val target = mediaDetail.value?.positionInSeconds?.let { position ->
                    if (position > 0) {
                        SmartNavigationTarget.toPlayer(it, null, mediaDetail.value?.positionInSeconds ?: 0)
                    } else SmartNavigationTarget.toPlayer(it, null)
                } ?: SmartNavigationTarget.toPlayer(it, null)
                target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
                navigator.navigate(target)
            }
            (v.context as? MainActivity)?.onBackPressed()
        }
    }

    fun onRestartButtonClicked(v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) restart button selected")
        ifCheckIsAdultContentLockedInvokeBlock(v) {
            val target = SmartNavigationTarget.toPlayer(_mediaReference.value!!, null, 0)
            target.extras = SmartDataValue.from(PlayerLocation.FOREGROUND.name)
            Navigator.getInstance().navigate(target)
            (v.context as? MainActivity)?.onBackPressed()
        }
    }

    private fun ifCheckIsAdultContentLockedInvokeBlock(v: View, doOnSuccess: () -> Unit) {
        _mediaReference.value ?: return
        mediaDetail.value ?: return
        val mediaItem = SmartMediaItem(
            _mediaReference.value!!,
            mediaDetail.value!!.type
        ).apply {
            extras = mediaDetail.value!!.extras
            restrictions = mediaDetail.value!!.restrictions
        }

        RestrictionChecker.checkRestrictions(
                context = v.context,
                mediaItem = mediaItem,
                doOnSuccess = doOnSuccess
        )
    }

    fun onDeleteButtonClicked(@Suppress("UNUSED_PARAMETER") v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) delete recording button selected")
        _startStopRecordingClicked.value = Unit
    }

    fun onStartButtonClicked(@Suppress("UNUSED_PARAMETER") v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) start recording button selected")
        _startStopRecordingClicked.value = Unit
    }

    fun onStopButtonClicked(@Suppress("UNUSED_PARAMETER") v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) stop recording button selected")
        _startStopRecordingClicked.value = Unit
    }

    fun onMoreKeyEvent(@Suppress("UNUSED_PARAMETER") v: View, keyCode: Int, keyEvent: KeyEvent): Boolean {
        if (_shownMore.value == true) {
            when (keyCode) {
                KeyEvent.KEYCODE_DPAD_UP -> {
                    scrollUp.call()
                    return true
                }
                KeyEvent.KEYCODE_DPAD_DOWN -> {
                    scrollDown.call()
                    return true
                }
            }
        } else if (rowViewVisibility.value == View.VISIBLE) {
            if (keyEvent.action == KeyEvent.ACTION_DOWN) {
                when (keyCode) {
                    KeyEvent.KEYCODE_DPAD_DOWN -> {
                        scrollDown.call()
                        return true
                    }
                }
            }
        }
        return false
    }

    fun onShowMoreButtonClicked(@Suppress("UNUSED_PARAMETER") v: View) {
        Timber.tag(OselToggleableLogger.TAG_UI_LOG).d("Detail (ID:${getIdForLog()}) show more button selected")
        setShownMore(_shownMore.value != true)
    }

    private fun getIdForLog(): String {
        return Flavor().getLogInfoFromSmartMediaReference(_mediaReference.value)
    }

    fun onPause() { if (::mTimer.isInitialized) mTimer.cancel() }

    internal fun getTitle():String = mediaDetail.value?.title ?: ""

    fun checkBroadcastStarted() = mediaDetail.value?.startDate?.isBeforeNow
}