package com.twentyfouri.tvlauncher.utils

import android.net.Uri
import android.text.TextUtils
import com.google.android.exoplayer2.C
import com.google.android.exoplayer2.upstream.*
import com.google.android.exoplayer2.upstream.HttpDataSource.*
import com.google.android.exoplayer2.util.Assertions
import com.google.android.exoplayer2.util.Util
import com.google.common.base.Predicate
import com.twentyfouri.smartexoplayer.httpoverrides.HttpDataSourceFactoryProvider
import com.twentyfouri.tvlauncher.common.provider.TimeProvider
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_ERROR_LOG
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger.Companion.TAG_PLAYER_STREAM_LOG
import timber.log.Timber
import java.io.*
import java.net.HttpURLConnection
import java.net.NoRouteToHostException
import java.net.ProtocolException
import java.net.URL
import java.util.regex.Matcher
import java.util.regex.Pattern
import java.util.zip.GZIPInputStream

/**
 * This is modified copy of com.google.android.exoplayer2.upstream.DefaultHttpDataSource version 2.13.3
 * because it cannot be extended and we need to log information about downloaded data
 */
class LoggingDefaultHttpDataSource private constructor(
    private val userAgent: String?,
    private val connectTimeoutMillis: Int,
    private val readTimeoutMillis: Int,
    private val allowCrossProtocolRedirects: Boolean,
    private val defaultRequestProperties: RequestProperties?,
    private var contentTypePredicate: Predicate<String>?
) : BaseDataSource(true), HttpDataSource {

    class Factory : HttpDataSource.Factory {
        private val defaultRequestProperties = RequestProperties()
        private var transferListener: TransferListener? = null
        private var contentTypePredicate: Predicate<String>? = null
        private var userAgent: String? = null
        private var connectTimeoutMs: Int = DEFAULT_CONNECT_TIMEOUT_MILLIS
        private var readTimeoutMs: Int = DEFAULT_READ_TIMEOUT_MILLIS
        private var allowCrossProtocolRedirects = false

        private val loggingTransferListener = object : TransferListener {
            val logs: MutableList<Pair<Long, String>> = arrayListOf()
            //to save some space in the log url is cut on the last '/' in case it is same as last logged url
            //but for reading convenience full url is logged after some limit
            var lastBasePath: String = ""
            var shortenedLogCount = 0
            val shortenedLogLimit = 20

            //consider log also moment when chunk/manifest download was started, currently it is logged only after download is finished
            override fun onTransferInitializing(source: DataSource, dataSpec: DataSpec, isNetwork: Boolean) {
                //set saveCopy to true if interested in content of response
                //currently only manifest seems interesting but also drm or chunks (subtitles?) might be
                //adapt also logging part "ADAPT HERE"
                //(source as? LoggingDefaultHttpDataSource)?.saveCopy = dataSpec.uri.toString().contains("manifest.mpd", true)
                logs.add(Pair(TimeProvider.nowMs(), dataSpec.uri.toString()))
            }

            override fun onTransferStart(source: DataSource, dataSpec: DataSpec, isNetwork: Boolean) {
            }

            override fun onBytesTransferred(source: DataSource, dataSpec: DataSpec, isNetwork: Boolean, bytesTransferred: Int) {
            }

            override fun onTransferEnd(source: DataSource, dataSpec: DataSpec, isNetwork: Boolean) {
                logs.find { it.second == dataSpec.uri.toString() }?.let {
                    val cds = source as? LoggingDefaultHttpDataSource
                    val basePath = dataSpec.uri.toString().substringBeforeLast('/')
                    val from = if(basePath == lastBasePath && shortenedLogCount < shortenedLogLimit ) {
                        shortenedLogCount++
                        "〃/${dataSpec.uri.toString().substringAfterLast('/')}"
                    } else {
                        shortenedLogCount = 0
                        lastBasePath = basePath
                        dataSpec.uri.toString()
                    }
                    Timber.tag(TAG_PLAYER_STREAM_LOG).d("DONE in ${TimeProvider.nowMs() - it.first}ms, read ${cds?.bytesRead}/${cds?.bytesToRead} bytes from $from")
                    if (cds?.redirects?.isNotEmpty() == true) Timber.tag(TAG_PLAYER_STREAM_LOG).d("↑ redirected to ${cds.redirects.joinToString("\n")}")
                    logs.remove(it)
                }
                //Timber.tag(TAG_PLAYER_STREAM_LOG).d("queue size ${logs.size}") //might be useful to log how many chunks are currently downloading
            }
        }

        @Deprecated("Use {@link #setDefaultRequestProperties(Map)} instead. ")
        override fun getDefaultRequestProperties(): RequestProperties {
            return defaultRequestProperties
        }

        override fun setDefaultRequestProperties(defaultRequestProperties: Map<String, String>): Factory {
            this.defaultRequestProperties.clearAndSet(defaultRequestProperties)
            return this
        }

        fun setUserAgent(userAgent: String?): Factory {
            this.userAgent = userAgent
            return this
        }

        fun setConnectTimeoutMs(connectTimeoutMs: Int): Factory {
            this.connectTimeoutMs = connectTimeoutMs
            return this
        }

        fun setReadTimeoutMs(readTimeoutMs: Int): Factory {
            this.readTimeoutMs = readTimeoutMs
            return this
        }

        fun setAllowCrossProtocolRedirects(allowCrossProtocolRedirects: Boolean): Factory {
            this.allowCrossProtocolRedirects = allowCrossProtocolRedirects
            return this
        }

        fun setContentTypePredicate(contentTypePredicate: Predicate<String>?): Factory {
            this.contentTypePredicate = contentTypePredicate
            return this
        }

        fun setTransferListener(transferListener: TransferListener?): Factory {
            this.transferListener = transferListener
            return this
        }

        override fun createDataSource(): LoggingDefaultHttpDataSource {
            val dataSource = LoggingDefaultHttpDataSource(
                userAgent,
                connectTimeoutMs,
                readTimeoutMs,
                allowCrossProtocolRedirects,
                defaultRequestProperties,
                contentTypePredicate
            )
            if (transferListener != null) {
                dataSource.addTransferListener(transferListener!!)
            }
            dataSource.addTransferListener(loggingTransferListener)
            return dataSource
        }
    }

    private val requestProperties = RequestProperties()
    private var dataSpec: DataSpec? = null

    private var connection: HttpURLConnection? = null
    private var inputStream: InputStream? = null
    private var skipBuffer: ByteArray? = null
    private var opened = false
    private var responseCode = 0
    private var bytesToSkip: Long = 0
    private var bytesToRead: Long = 0
    private var bytesSkipped: Long = 0
    private var bytesRead: Long = 0

    val redirects: MutableList<String> = mutableListOf()
    private var copyStream: ByteArrayOutputStream? = null
    var saveCopy: Boolean = false

    override fun getUri(): Uri? {
        return if (connection == null) null else Uri.parse(connection!!.url.toString())
    }

    override fun getResponseCode(): Int {
        return if (connection == null || responseCode <= 0) -1 else responseCode
    }

    override fun getResponseHeaders(): Map<String, List<String>> {
        return if (connection == null) emptyMap() else connection!!.headerFields
    }

    override fun setRequestProperty(name: String, value: String) {
        Assertions.checkNotNull(name)
        Assertions.checkNotNull(value)
        requestProperties[name] = value
    }

    override fun clearRequestProperty(name: String) {
        Assertions.checkNotNull(name)
        requestProperties.remove(name)
    }

    override fun clearAllRequestProperties() {
        requestProperties.clear()
    }

    @Throws(HttpDataSourceException::class)
    override fun open(dataSpec: DataSpec): Long {
        this.dataSpec = dataSpec
        bytesRead = 0
        bytesSkipped = 0
        transferInitializing(dataSpec)
        if (saveCopy) copyStream = ByteArrayOutputStream()
        try {
            connection = makeConnection(dataSpec)
        } catch (e: IOException) {
            val message = e.message
            if (message != null && message.contains("cleartext http traffic.*not permitted.*", true)) {
                throw CleartextNotPermittedException(e, dataSpec)
            }
            throw HttpDataSourceException("Unable to connect", e, dataSpec, HttpDataSourceException.TYPE_OPEN)
        }
        val connection = connection
        val responseMessage: String
        try {
            responseCode = connection!!.responseCode
            responseMessage = connection.responseMessage
        } catch (e: IOException) {
            closeConnectionQuietly()
            throw HttpDataSourceException("Unable to connect", e, dataSpec, HttpDataSourceException.TYPE_OPEN)
        }

        // Check for a valid response code.
        if (responseCode < 200 || responseCode > 299) {
            val headers = connection.headerFields
            val errorStream = connection.errorStream
            val errorResponseBody: ByteArray = try {
                if (errorStream != null) Util.toByteArray(errorStream) else Util.EMPTY_BYTE_ARRAY
            } catch (e: IOException) {
                Util.EMPTY_BYTE_ARRAY
            }
            closeConnectionQuietly()
            val exception = InvalidResponseCodeException(responseCode, responseMessage, headers, dataSpec, errorResponseBody)
            if (responseCode == 416) {
                exception.initCause(DataSourceException(DataSourceException.POSITION_OUT_OF_RANGE))
            }
            throw exception
        }

        // Check for a valid content type.
        val contentType = connection.contentType
        if (contentTypePredicate != null && !contentTypePredicate!!.apply(contentType)) {
            closeConnectionQuietly()
            throw InvalidContentTypeException(contentType, dataSpec)
        }

        // If we requested a range starting from a non-zero position and received a 200 rather than a
        // 206, then the server does not support partial requests. We'll need to manually skip to the
        // requested position.
        bytesToSkip = if (responseCode == 200 && dataSpec.position != 0L) dataSpec.position else 0

        // Determine the length of the data to be read, after skipping.
        val isCompressed: Boolean = isCompressed(connection)
        bytesToRead = if (!isCompressed) {
            if (dataSpec.length != C.LENGTH_UNSET.toLong()) {
                dataSpec.length
            } else {
                val contentLength: Long = getContentLength(connection)
                if (contentLength != C.LENGTH_UNSET.toLong()) contentLength - bytesToSkip else C.LENGTH_UNSET.toLong()
            }
        } else {
            // Gzip is enabled. If the server opts to use gzip then the content length in the response
            // will be that of the compressed data, which isn't what we want. Always use the dataSpec
            // length in this case.
            dataSpec.length
        }
        inputStream = try {
            if (isCompressed) {
                GZIPInputStream(connection.inputStream)
            } else {
                connection.inputStream
            }
        } catch (e: IOException) {
            closeConnectionQuietly()
            throw HttpDataSourceException(e, dataSpec, HttpDataSourceException.TYPE_OPEN)
        }
        opened = true
        transferStarted(dataSpec)
        return bytesToRead
    }

    @Throws(HttpDataSourceException::class)
    override fun read(buffer: ByteArray, offset: Int, readLength: Int): Int {
        try {
            skipInternal()
            return readInternal(buffer, offset, readLength)
        } catch (e: IOException) {
            throw HttpDataSourceException(
                e, Util.castNonNull(dataSpec), HttpDataSourceException.TYPE_READ
            )
        }
    }

    @Throws(HttpDataSourceException::class)
    override fun close() {
        try {
            val inputStream = inputStream
            if (inputStream != null) {
                maybeTerminateInputStream(connection, bytesRemaining())
                try {
                    copyStream?.let {
                        //ADAPT HERE
                        Timber.tag(TAG_PLAYER_STREAM_LOG).d("MANIFEST ${it.toString(Charsets.UTF_8.name())}")
                        //Timber.tag(TAG_PLAYER_STREAM_LOG).d("copy have ${it.size()}") //useful to check if really whole content was caught
                        it.close()
                    }
                    inputStream.close()
                } catch (e: IOException) {
                    throw HttpDataSourceException(e, Util.castNonNull(dataSpec), HttpDataSourceException.TYPE_CLOSE)
                }
            }
        } finally {
            inputStream = null
            closeConnectionQuietly()
            if (opened) {
                opened = false
                transferEnded()
            }
        }
    }

    private fun bytesRemaining(): Long {
        return if (bytesToRead == C.LENGTH_UNSET.toLong()) bytesToRead else bytesToRead - bytesRead
    }

    @Throws(IOException::class)
    private fun makeConnection(dataSpec: DataSpec): HttpURLConnection {
        var url = URL(dataSpec.uri.toString())
        @DataSpec.HttpMethod var httpMethod = dataSpec.httpMethod
        var httpBody = dataSpec.httpBody
        val position = dataSpec.position
        val length = dataSpec.length
        val allowGzip = dataSpec.isFlagSet(DataSpec.FLAG_ALLOW_GZIP)

        // We need to handle redirects ourselves to allow cross-protocol redirects
        // and also be able to log redirect steps
        var redirectCount = 0
        redirects.clear()
        while (redirectCount++ <= MAX_REDIRECTS) {
            val connection = makeConnection(url, httpMethod, httpBody, position, length, allowGzip, dataSpec.httpRequestHeaders)
            val responseCode = connection.responseCode
            val location = connection.getHeaderField("Location")
            if ((httpMethod == DataSpec.HTTP_METHOD_GET || httpMethod == DataSpec.HTTP_METHOD_HEAD)
                && (responseCode == HttpURLConnection.HTTP_MULT_CHOICE || responseCode == HttpURLConnection.HTTP_MOVED_PERM || responseCode == HttpURLConnection.HTTP_MOVED_TEMP || responseCode == HttpURLConnection.HTTP_SEE_OTHER || responseCode == HTTP_STATUS_TEMPORARY_REDIRECT || responseCode == HTTP_STATUS_PERMANENT_REDIRECT)
            ) {
                connection.disconnect()
                redirects.add(location.toString())
                url = handleRedirect(url, location, allowCrossProtocolRedirects)
            } else if (httpMethod == DataSpec.HTTP_METHOD_POST
                && (responseCode == HttpURLConnection.HTTP_MULT_CHOICE || responseCode == HttpURLConnection.HTTP_MOVED_PERM || responseCode == HttpURLConnection.HTTP_MOVED_TEMP || responseCode == HttpURLConnection.HTTP_SEE_OTHER)
            ) {
                // POST request follows the redirect and is transformed into a GET request.
                connection.disconnect()
                redirects.add(location.toString())
                httpMethod = DataSpec.HTTP_METHOD_GET
                httpBody = null
                url = handleRedirect(url, location, allowCrossProtocolRedirects)
            } else {
                return connection
            }
        }
        throw NoRouteToHostException("Too many redirects: $redirectCount")
    }

    @Throws(IOException::class)
    private fun makeConnection(
        url: URL,
        @DataSpec.HttpMethod httpMethod: Int,
        httpBody: ByteArray?,
        position: Long,
        length: Long,
        allowGzip: Boolean,
        requestParameters: Map<String, String>
    ): HttpURLConnection {
        val connection = openConnection(url)
        connection.connectTimeout = connectTimeoutMillis
        connection.readTimeout = readTimeoutMillis
        val requestHeaders: MutableMap<String, String> = HashMap()
        if (defaultRequestProperties != null) {
            requestHeaders.putAll(defaultRequestProperties.snapshot)
        }
        requestHeaders.putAll(requestProperties.snapshot)
        requestHeaders.putAll(requestParameters)
        for (property: Map.Entry<String, String> in requestHeaders.entries) {
            connection.setRequestProperty(property.key, property.value)
        }
        if (!(position == 0L && length == C.LENGTH_UNSET.toLong())) {
            var rangeRequest = "bytes=$position-"
            if (length != C.LENGTH_UNSET.toLong()) {
                rangeRequest += position + length - 1
            }
            connection.setRequestProperty("Range", rangeRequest)
        }
        if (userAgent != null) {
            connection.setRequestProperty("User-Agent", userAgent)
        }
        connection.setRequestProperty("Accept-Encoding", if (allowGzip) "gzip" else "identity")
        connection.instanceFollowRedirects = false
        connection.doOutput = httpBody != null
        connection.requestMethod = DataSpec.getStringForHttpMethod(httpMethod)
        if (httpBody != null) {
            connection.setFixedLengthStreamingMode(httpBody.size)
            connection.connect()
            val os = connection.outputStream
            os.write(httpBody)
            os.close()
        } else {
            connection.connect()
        }
        return connection
    }

    @Throws(IOException::class)
    fun openConnection(url: URL): HttpURLConnection {
        return url.openConnection() as HttpURLConnection
    }

    @Throws(IOException::class)
    private fun skipInternal() {
        if (bytesSkipped == bytesToSkip) {
            return
        }
        if (skipBuffer == null) {
            skipBuffer = ByteArray(4096)
        }
        while (bytesSkipped != bytesToSkip) {
            val readLength = (bytesToSkip - bytesSkipped).coerceAtMost(skipBuffer!!.size.toLong()).toInt()
            val read = Util.castNonNull(inputStream).read(skipBuffer, 0, readLength)
            if (Thread.currentThread().isInterrupted) {
                throw InterruptedIOException()
            }
            if (read == -1) {
                throw EOFException()
            }
            bytesSkipped += read.toLong()
            bytesTransferred(read)
        }
    }

    @Throws(IOException::class)
    private fun readInternal(buffer: ByteArray, offset: Int, readLength: Int): Int {
        var readLengthI = readLength
        if (readLengthI == 0) {
            return 0
        }
        if (bytesToRead != C.LENGTH_UNSET.toLong()) {
            val bytesRemaining = bytesToRead - bytesRead
            if (bytesRemaining == 0L) {
                return C.RESULT_END_OF_INPUT
            }
            readLengthI = readLength.toLong().coerceAtMost(bytesRemaining).toInt()
        }
        val read = Util.castNonNull(inputStream).read(buffer, offset, readLengthI)
        copyStream?.write(buffer, 0, read)
        if (read == -1) {
            if (bytesToRead != C.LENGTH_UNSET.toLong()) {
                // End of stream reached having not read sufficient data.
                throw EOFException()
            }
            return C.RESULT_END_OF_INPUT
        }
        bytesRead += read.toLong()
        bytesTransferred(read)
        return read
    }

    private fun closeConnectionQuietly() {
        if (connection != null) {
            try {
                connection!!.disconnect()
            } catch (e: Exception) {
                Timber.tag(TAG_PLAYER_ERROR_LOG).e( e,"Unexpected error while disconnecting" )
            }
            connection = null
        }
    }

    companion object {
        const val DEFAULT_CONNECT_TIMEOUT_MILLIS = 8 * 1000
        const val DEFAULT_READ_TIMEOUT_MILLIS = 8 * 1000
        private const val MAX_REDIRECTS = 20 // Same limit as okhttp.
        private const val HTTP_STATUS_TEMPORARY_REDIRECT = 307
        private const val HTTP_STATUS_PERMANENT_REDIRECT = 308
        private const val MAX_BYTES_TO_DRAIN: Long = 2048
        private val CONTENT_RANGE_HEADER = Pattern.compile("^bytes (\\d+)-(\\d+)/(\\d+)$")

        val httpDataSourceFactoryProvider = object : HttpDataSourceFactoryProvider {
            override fun getHttpDataSourceFactory(userAgent: String, listener: TransferListener?): HttpDataSource.Factory {
                return Factory().apply {
                    this.setUserAgent(userAgent)
                    this.setTransferListener(listener)
                }
            }
        }

        @Throws(IOException::class)
        private fun handleRedirect(originalUrl: URL, location: String?, allowCrossProtocolRedirects: Boolean): URL {
            if (location == null) {
                throw ProtocolException("Null location redirect")
            }
            // Form the new url.
            val url = URL(originalUrl, location)
            // Check that the protocol of the new url is supported.
            val protocol = url.protocol
            if ("https" != protocol && "http" != protocol) {
                throw ProtocolException("Unsupported protocol redirect: $protocol")
            }
            // Currently this method is only called if allowCrossProtocolRedirects is true, and so the code
            // below isn't required. If we ever decide to handle redirects ourselves when cross-protocol
            // redirects are disabled, we'll need to uncomment this block of code.
            if (!allowCrossProtocolRedirects && !protocol.equals(originalUrl.protocol)) {
                throw ProtocolException("Disallowed cross-protocol redirect (${originalUrl.protocol} to $protocol)")
            }
            return url
        }

        private fun getContentLength(connection: HttpURLConnection): Long {
            var contentLength = C.LENGTH_UNSET.toLong()
            val contentLengthHeader = connection.getHeaderField("Content-Length")
            if (!TextUtils.isEmpty(contentLengthHeader)) {
                try {
                    contentLength = contentLengthHeader.toLong()
                } catch (e: NumberFormatException) {
                    Timber.tag(TAG_PLAYER_ERROR_LOG).e("Unexpected Content-Length [$contentLengthHeader]")
                }
            }
            val contentRangeHeader = connection.getHeaderField("Content-Range")
            if (!TextUtils.isEmpty(contentRangeHeader)) {
                val matcher: Matcher = CONTENT_RANGE_HEADER.matcher(contentRangeHeader)
                if (matcher.find()) {
                    try {
                        val contentLengthFromRange = (Assertions.checkNotNull(matcher.group(2))
                            .toLong() - Assertions.checkNotNull(matcher.group(1)).toLong()
                                + 1)
                        if (contentLength < 0) {
                            // Some proxy servers strip the Content-Length header. Fall back to the length
                            // calculated here in this case.
                            contentLength = contentLengthFromRange
                        } else if (contentLength != contentLengthFromRange) {
                            // If there is a discrepancy between the Content-Length and Content-Range headers,
                            // assume the one with the larger value is correct. We have seen cases where carrier
                            // change one of them to reduce the size of a request, but it is unlikely anybody would
                            // increase it.
                            Timber.tag(TAG_PLAYER_ERROR_LOG).w("Inconsistent headers [$contentLengthHeader] [$contentRangeHeader]")
                            contentLength = contentLength.coerceAtLeast(contentLengthFromRange)
                        }
                    } catch (e: NumberFormatException) {
                        Timber.tag(TAG_PLAYER_ERROR_LOG).e("Unexpected Content-Range [$contentRangeHeader]")
                    }
                }
            }
            return contentLength
        }

        private fun maybeTerminateInputStream(connection: HttpURLConnection?, bytesRemaining: Long) {
            if ((connection == null) || (Util.SDK_INT < 19) || (Util.SDK_INT > 20)) {
                return
            }
            try {
                val inputStream = connection.inputStream
                if (bytesRemaining == C.LENGTH_UNSET.toLong()) {
                    // If the input stream has already ended, do nothing. The socket may be re-used.
                    if (inputStream.read() == -1) {
                        return
                    }
                } else if (bytesRemaining <= MAX_BYTES_TO_DRAIN) {
                    // There isn't much data left. Prefer to allow it to drain, which may allow the socket to be
                    // re-used.
                    return
                }
                val className = inputStream.javaClass.name
                if ((("com.android.okhttp.internal.http.HttpTransport\$ChunkedInputStream" == className)
                            || ("com.android.okhttp.internal.http.HttpTransport\$FixedLengthInputStream" == className))
                ) {
                    val superclass: Class<*>? = inputStream.javaClass.superclass
                    val unexpectedEndOfInput = Assertions.checkNotNull(superclass).getDeclaredMethod("unexpectedEndOfInput")
                    unexpectedEndOfInput.isAccessible = true
                    unexpectedEndOfInput.invoke(inputStream)
                }
            } catch (e: Exception) {
                // If an IOException then the connection didn't ever have an input stream, or it was closed
                // already. If another type of exception then something went wrong, most likely the device
                // isn't using okhttp.
            }
        }

        private fun isCompressed(connection: HttpURLConnection): Boolean {
            val contentEncoding = connection.getHeaderField("Content-Encoding")
            return "gzip".equals(contentEncoding, ignoreCase = true)
        }
    }
}