package com.twentyfouri.tvlauncher.utils

import androidx.lifecycle.LiveData
import androidx.lifecycle.MediatorLiveData
import androidx.lifecycle.Observer
import com.twentyfouri.tvlauncher.common.extensions.ifTrue

object CombinationTransformations {
    fun <A, B, C> combineNullable(
        data1: LiveData<A?>,
        data2: LiveData<B?>,
        combineFun: (data11: A?, data22: B?) -> C
    ): MediatorLiveData<C> {
        val result = CombinedNullableLiveData(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        return result
    }

    fun <A, B, C> zipCombineNullable(
            data1: LiveData<A?>,
            data2: LiveData<B?>,
            combineFun: (data11: A?, data22: B?) -> C
    ): MediatorLiveData<C> {
        val result = ZipCombinedNullableLiveData(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        return result
    }

    fun <A : Any, B, C> combineHalfNullable(
            data1: LiveData<A>,
            data2: LiveData<B?>,
            combineFun: (data11: A, data22: B?) -> C
    ): MediatorLiveData<C> {
        val result = CombinedHalfNullableLiveData(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        return result
    }

    fun <A, B, C, X> combineNullable(
        data1: LiveData<A?>,
        data2: LiveData<B?>,
        data3: LiveData<C?>,
        combineFun: (data11: A?, data22: B?, data33: C?) -> X
    ): MediatorLiveData<X> {
        val result = CombinedNullableLiveDataTriple(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        result.addSource3(data3)
        return result
    }

    fun <A: Any, B: Any, C: Any, X> combineNonNullable(
        data1: LiveData<A>,
        data2: LiveData<B>,
        data3: LiveData<C>,
        combineFun: (data11: A, data22: B, data33: C) -> X
    ): MediatorLiveData<X> {
        val result = CombinedNonNullableLiveDataTriple(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        result.addSource3(data3)
        return result
    }

    fun <A1: Any, B1: Any, C1> combineNonNullable(
        data1: LiveData<A1>,
        data2: LiveData<B1>,
        combineFun: (data11: A1, data22: B1) -> C1
    ): MediatorLiveData<C1> {
        val result = CombinedNonNullableLiveData(combineFun)
        result.addSource1(data1)
        result.addSource2(data2)
        return result
    }

    fun <D, E, F> map(
        data1: LiveData<D>,
        fun1: (data11: D) -> F,
        data2: LiveData<E>,
        fun2: (data22: E) -> F
    ): LiveData<F> {
        val result = MediatorLiveData<F>()
        result.addSource(data1) { result.value = fun1(it) }
        result.addSource(data2) { result.value = fun2(it) }
        return result
    }

    fun <G, H, I, J> map(
        data1: LiveData<G>,
        fun1: (data11: G) -> J,
        data2: LiveData<H>,
        fun2: (data22: H) -> J,
        data3: LiveData<I>,
        fun3: (data33: I) -> J,
        prerequisites: List<LiveData<out Any>> = emptyList()
    ): LiveData<J> {
        val result = MediatorLiveData<J>()
        val prerequisiteLiveData = PrerequisiteLiveData(prerequisites)
        result.addSource(prerequisiteLiveData) {
            result.removeSource(prerequisiteLiveData)
            result.addSource(data1) { result.value = fun1(it) }
            result.addSource(data2) { result.value = fun2(it) }
            result.addSource(data3) { result.value = fun3(it) }
        }
        return result
    }

    fun <I1, I2, I3, I4, O> map(
        data1: LiveData<I1>,
        fun1: (data11: I1) -> O,
        data2: LiveData<I2>,
        fun2: (data22: I2) -> O,
        data3: LiveData<I3>,
        fun3: (data33: I3) -> O,
        data4: LiveData<I4>,
        fun4: (data33: I4) -> O,
        prerequisites: List<LiveData<out Any>> = emptyList()
    ): LiveData<O> {
        val result = MediatorLiveData<O>()
        val prerequisiteLiveData = PrerequisiteLiveData(prerequisites)
        result.addSource(prerequisiteLiveData) {
            result.removeSource(prerequisiteLiveData)
            result.addSource(data1) { result.value = fun1(it) }
            result.addSource(data2) { result.value = fun2(it) }
            result.addSource(data3) { result.value = fun3(it) }
            result.addSource(data4) { result.value = fun4(it) }
        }
        return result
    }

    fun <I1, I2, I3, I4, I5, O> map(
        data1: LiveData<I1>,
        fun1: (data11: I1) -> O,
        data2: LiveData<I2>,
        fun2: (data22: I2) -> O,
        data3: LiveData<I3>,
        fun3: (data33: I3) -> O,
        data4: LiveData<I4>,
        fun4: (data44: I4) -> O,
        data5: LiveData<I5>,
        fun5: (data55: I5) -> O,
        prerequisites: List<LiveData<out Any>> = emptyList()
    ): LiveData<O> {
        val result = MediatorLiveData<O>()
        val prerequisiteLiveData = PrerequisiteLiveData(prerequisites)
        result.addSource(prerequisiteLiveData) {
            result.removeSource(prerequisiteLiveData)
            result.addSource(data1) { result.value = fun1(it) }
            result.addSource(data2) { result.value = fun2(it) }
            result.addSource(data3) { result.value = fun3(it) }
            result.addSource(data4) { result.value = fun4(it) }
            result.addSource(data5) { result.value = fun5(it) }
        }
        return result
    }

    fun <K> map(
        fun1: () -> K,
        prerequisites: List<LiveData<out Any>> = emptyList()
    ): LiveData<K> {
        val result = MediatorLiveData<K>()
        val prerequisiteLiveData = PrerequisiteLiveData(prerequisites)
        result.addSource(prerequisiteLiveData) {
            result.removeSource(prerequisiteLiveData)
            result.value = fun1()
        }
        return result
    }

    fun <X, Y> switchMap(
        source: LiveData<X>,
        switchMapFunction: (X) -> LiveData<Y>
    ): LiveData<Y?> {
        val result = MediatorLiveData<Y?>()
        result.addSource(source, object : Observer<X> {
            var mSource: LiveData<Y>? = null
            override fun onChanged(x: X) {
                val newLiveData = switchMapFunction(x)
                if (mSource === newLiveData) {
                    return
                }
                if (mSource != null) {
                    result.removeSource(mSource!!)
                }
                mSource = newLiveData
                if (mSource != null) {
                    result.addSource(mSource!!) { y -> result.value = y }
                }
            }
        })
        return result
    }

    private class CombinedNullableLiveData<K, L, M>(combineFun: (data1: K?, data2: L?) -> M) : MediatorLiveData<M>() {
        private var data1: K? = null
        private var data2: L? = null
        private val combine = combineFun

        fun addSource1(source: LiveData<K?>) {
            super.addSource(source) {
                data1 = it
                value = combine(data1, data2)
            }
        }

        fun addSource2(source: LiveData<L?>) {
            super.addSource(source) {
                data2 = it
                value = combine(data1, data2)
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }


    private class ZipCombinedNullableLiveData<K, L, M>(combineFun: (data1: K?, data2: L?) -> M) : MediatorLiveData<M>() {
        private var data1: K? = null
        private var data2: L? = null
        private val combine = combineFun

        fun addSource1(source: LiveData<K?>) {
            super.addSource(source) {
                data1 = it
                update()
            }
        }

        fun addSource2(source: LiveData<L?>) {
            super.addSource(source) {
                data2 = it
                update()
            }
        }

        fun update() {
            val localData1 = data1
            val localData2 = data2
            if (localData1 != null && localData2 != null) {
                this.value = combine(data1, data2)
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }

    private class CombinedHalfNullableLiveData<K1: Any, L1, M1>(combineFun: (data1: K1, data2: L1?) -> M1) : MediatorLiveData<M1>() {
        private lateinit var data1: K1
        private var data2: L1? = null
        private val combine = combineFun

        fun addSource1(source: LiveData<K1>) {
            super.addSource(source) {
                data1 = it
                value = combine(data1, data2)
            }
        }

        fun addSource2(source: LiveData<L1?>) {
            super.addSource(source) {
                data2 = it
                ::data1.isInitialized.ifTrue { value = combine(data1, data2) }
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }

    private class CombinedNonNullableLiveData<K1: Any, L1: Any, M1>(combineFun: (data1: K1, data2: L1) -> M1) : MediatorLiveData<M1>() {
        private lateinit var data1: K1
        private lateinit var data2: L1
        private val combine = combineFun

        fun addSource1(source: LiveData<K1>) {
            super.addSource(source) {
                data1 = it
                ::data2.isInitialized.ifTrue { value = combine(data1, data2) }
            }
        }

        fun addSource2(source: LiveData<L1>) {
            super.addSource(source) {
                data2 = it
                ::data1.isInitialized.ifTrue { value = combine(data1, data2) }
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }

    private class CombinedNonNullableLiveDataTriple<K: Any, L: Any, M: Any, X>(combineFun: (data1: K, data2: L, data3: M) -> X) : MediatorLiveData<X>() {
        private lateinit var data1: K
        private lateinit var data2: L
        private lateinit var data3: M
        private val combine = combineFun

        fun addSource1(source: LiveData<K>) {
            super.addSource(source) {
                data1 = it
                (::data2.isInitialized && ::data3.isInitialized).ifTrue { value = combine(data1, data2, data3) }
            }
        }

        fun addSource2(source: LiveData<L>) {
            super.addSource(source) {
                data2 = it
                (::data1.isInitialized && ::data3.isInitialized).ifTrue { value = combine(data1, data2, data3) }
            }
        }

        fun addSource3(source: LiveData<M>) {
            super.addSource(source) {
                data3 = it
                (::data1.isInitialized && ::data2.isInitialized).ifTrue { value = combine(data1, data2, data3) }
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }

    private class CombinedNullableLiveDataTriple<K, L, M, X>(combineFun: (data1: K?, data2: L?, data3: M?) -> X) : MediatorLiveData<X>() {
        private var data1: K? = null
        private var data2: L? = null
        private var data3: M? = null
        private val combine = combineFun

        fun addSource1(source: LiveData<K?>) {
            super.addSource(source) {
                data1 = it
                value = combine(data1, data2, data3)
            }
        }

        fun addSource2(source: LiveData<L?>) {
            super.addSource(source) {
                data2 = it
                value = combine(data1, data2, data3)
            }
        }

        fun addSource3(source: LiveData<M?>) {
            super.addSource(source) {
                data3 = it
                value = combine(data1, data2, data3)
            }
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }

    open class PrerequisiteLiveData(val prerequisites: List<LiveData<out Any>>) : MediatorLiveData<Boolean>() {

        init {
            prerequisites.forEach{
                super.addSource(it) {
                    checkPrerequisites()
                }
            }
            if (prerequisites.isEmpty()) value = true
        }

        private fun checkPrerequisites() {
            val prepared = prerequisites.all { it.value != null }
            if (prepared) value = true
        }

        override fun <N : Any?> addSource(source: LiveData<N>, onChanged: Observer<in N>) {
            throw UnsupportedOperationException()
        }

        override fun <O : Any?> removeSource(toRemote: LiveData<O>) {
            throw UnsupportedOperationException()
        }
    }
}