package com.twentyfouri.tvlauncher.utils

import android.app.AlertDialog
import android.content.Context
import android.content.Intent
import android.content.pm.PackageManager
import android.util.Log
import androidx.core.content.FileProvider
import com.twentyfouri.tvlauncher.common.extensions.ifElse
import com.twentyfouri.tvlauncher.common.extensions.ifFalse
import com.twentyfouri.tvlauncher.common.extensions.ifTrue
import com.twentyfouri.tvlauncher.BuildConfig
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import timber.log.Timber
import java.io.File
import java.io.FileOutputStream
import java.io.InputStream

object AppManager {

    private val TAG: String = AppManager::class.java.simpleName
    private var printLogs = false

    /**
     * Install APK using PackageInstaller
     * @param context Context necessary to start the intent for installing the app.
     * @param apkFile  File object of APK.
     */
    private fun installAPK(context: Context, apkFile: File) {
        if (printLogs) Timber.d("installAPK -> Installing apk ${apkFile.absolutePath}")
        val installIntent = Intent("android.intent.action.VIEW")
        installIntent.addCategory("android.intent.category.DEFAULT")
        installIntent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
        installIntent.addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
        installIntent.putExtra(Intent.EXTRA_NOT_UNKNOWN_SOURCE, true)
        installIntent.setDataAndType(
            FileProvider.getUriForFile(context, context.applicationContext.packageName + ".FileProvider", apkFile), "application/vnd.android.package-archive"
        )
        context.startActivity(installIntent)
    }

    /**
     * Install APK using PackageInstaller
     * @param context Context necessary to start the intent for installing the app.
     * @param apkPath File path of APK.
     * @param appName Name of the app to install.
     */
    private fun installAPK(context: Context, apkPath: String, appName: String) {
        if (printLogs) Timber.d("installAPK -> Installing apk $apkPath with name: $appName.")
        val assetManager = context.assets
        val inputStream: InputStream?
        val outputStream: FileOutputStream?
        try {
            inputStream = assetManager.open(apkPath)
            val apkDataPath = "${context.filesDir.path}/${appName}.apk"
            if (printLogs) Timber.d("installAPK -> Output file saved to $apkDataPath.")
            outputStream = FileOutputStream(apkDataPath)
            val buffer = ByteArray(1024)
            var read: Int

            while (true) {
                read = inputStream.read(buffer)
                if (read != -1) {
                    outputStream.write(buffer, 0, read)
                } else break
            }

            inputStream.close()
            outputStream.flush()
            outputStream.close()

            installAPK(context, File(apkDataPath))
        } catch (exception: Exception) {
            if (printLogs) Timber.d("installAPK -> Exception: $exception")
        }
    }

    /**
     * Install batch of APKs.
     * @param context: Context necessary for checking if apps are already installed and to get some params from resources.
     * @param apkPaths: HashMap <String, String> of package names and its respective apk file names.
     * @param checkFlavor: If we want to check if the package that we want to install is the same flavor as launcher/app.
     */
    private fun installAPKBatch(context: Context, apkPaths: HashMap<String, String>, checkFlavor: Boolean = false) {
        if (printLogs) Timber.d("installAPKBatch -> Installing a batch of APKs.")
        apkPaths.forEach { (packageName, apkPath) ->
            if (isPackageInstalled(packageName, context.packageManager, checkFlavor).not()) {
                installAPK(context, apkPath, packageName)
            } else {
                if (printLogs) Timber.d("installAPKBatch -> App is already installed.")
            }
        }
    }

    /**
     * Install searching utilities.
     * @param context: Context necessary to get package names and apk file names from resources.
     * @param onMessageRead: Block to execute when the user reads the installation message.
     */
    fun installSearchingUtilities(context: Context, onMessageRead: () -> Unit) {
        if (printLogs) Timber.d("installSearchingUtilities -> Installing search utilities.")
        if (BuildConfig.IS_DEBUG && Flavor().getSearchableDelegate() != null) {
            if (printLogs) Timber.d("installSearchingUtilities -> Installing search batch of APKs for flavor: ${BuildConfig.FLAVOR}.")
            var showDialog = false
            Flavor().getSearchableDelegate()?.getSearchableAppsInfo(context.assets)?.let { searchProxiesInfo ->
                searchProxiesInfo.forEach { (packageName, _) ->
                    val searchProxyIsInstalled: Boolean = isPackageInstalled(packageName, context.packageManager, true)
                    searchProxyIsInstalled.ifFalse { showDialog = true }
                }

                showDialog.ifTrue { showSearchAppInstallationDialog(context, searchProxiesInfo, onMessageRead) }.ifElse { onMessageRead() }
            }
        } else onMessageRead()
    }

    /**
     *  Method to show a dialog that lets the user decide if wants to install search proxies.
     *  @param context: Context necessary to build the alert dialog.
     *  @param searchProxiesInfo: HashMap <String, String> of package names and its respective apk file names.
     *  @param onMessageRead: Block to execute when the user reads the installation message.
     */
    private fun showSearchAppInstallationDialog(context: Context, searchProxiesInfo: HashMap<String, String>, onMessageRead: () -> Unit) {
        if (printLogs) Timber.d("showSearchAppInstallationDialog -> Showing dialog.")
        AlertDialog.Builder(context)
            .setTitle(context.resources.getString(R.string.search_proxies_dialog_title))
            .setMessage(context.resources.getString(R.string.search_proxies_dialog_message))
            .setPositiveButton(context.resources.getString(R.string.search_proxies_possitive_button)) { dialog, _ ->
                installAPKBatch(context, searchProxiesInfo, true)
                dialog?.dismiss()
            }
            .setNegativeButton(context.resources.getString(R.string.search_proxies_negative_button)) { dialog, _ ->
                dialog?.dismiss()
            }
            .setOnDismissListener {
                onMessageRead()
            }
            .show()
    }

    /**
     * Check if an specific package is installed.
     * @param packageName: Name of the package we want to check if is installed.
     * @param packageManager: Package manager necessary to check if the package is installed.
     * @param checkFlavor: If we want to check if the package that we want to install is the same flavor as launcher/app.
     * @return Boolean: If the package is already installed or not.
     */
    private fun isPackageInstalled(packageName: String, packageManager: PackageManager, checkFlavor: Boolean = false): Boolean {
        if (printLogs) Timber.d("isPackageInstalled -> Checking if package with name: $packageName is already installed.")
        return try {
            val applicationInfo = packageManager.getApplicationInfo(packageName, 0)
            val packageIsInstalled = applicationInfo.enabled
            if (printLogs) Timber.d("isPackageInstalled -> It is installed? $packageIsInstalled.")
            return if (checkFlavor) {
                val packageInfo = packageManager.getPackageInfo(packageName, 0)
                val isCurrentFlavor = packageInfo.versionName.contains(BuildConfig.FLAVOR_product)
                if (printLogs) Timber.d("isPackageInstalled -> ${packageInfo.versionName} is ${BuildConfig.FLAVOR_product}? $isCurrentFlavor.")

                packageIsInstalled && isCurrentFlavor
            } else {
                packageIsInstalled
            }
        } catch (e: PackageManager.NameNotFoundException) {
            if (printLogs) Timber.d("isPackageInstalled -> Exception: ${e.message}")
            false
        }
    }
}