package com.twentyfouri.tvlauncher.ui

import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.databinding.DataBindingUtil
import androidx.fragment.app.FragmentActivity
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.Observer
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchaseState
import com.twentyfouri.tvlauncher.common.utils.NavigatorCommon
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogCodes
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogFragment
import com.twentyfouri.tvlauncher.common.ui.messagedialog.MessageDialogModel
import com.twentyfouri.tvlauncher.databinding.FragmentSubscriptionsBinding
import com.twentyfouri.tvlauncher.utils.RemoteControlState
import com.twentyfouri.tvlauncher.viewmodels.SubscriptionsViewModel
import org.koin.androidx.viewmodel.ext.android.getViewModel

class SubscriptionsFragment: BaseFragment() {

    private var nullableBinding : FragmentSubscriptionsBinding? = null
    private val binding: FragmentSubscriptionsBinding get() = nullableBinding ?: throw IllegalStateException("trying to access uninitialized binding")

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentSubscriptionsBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@SubscriptionsFragment.viewLifecycleOwner
        nullableBinding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        binding.viewModel = getViewModel()

        val channelItem = arguments?.getSerializable(ARG_CHANNEL_ITEM) as? SmartMediaItem
        bind(channelItem, viewLifecycleOwner)
    }

    fun bind(subscriptionChannel: SmartMediaItem?, lifecycleOwner: LifecycleOwner) {
            binding.viewModel?.setSubscriptionChannel(subscriptionChannel)
            ((context as FragmentActivity) as MainActivity).remoteKeyManager.setRemoteControlState(RemoteControlState.PLAYER_BEHIND_SUBSCRIPTION)
            binding.offers.requestFocus()
            binding.viewModel?.subscriptionBought?.observe(lifecycleOwner, Observer {

                it?.let { purchaseResult ->
                    if (purchaseResult.state == SmartPurchaseState.OK) {
                        binding.viewModel?.setSubscriptionBought(null)
                        ((context as FragmentActivity) as MainActivity).refreshPlayerAfterSubscriptionBought()
                        showPurchaseSuccessDialog()
                    }

                    if (purchaseResult.state == SmartPurchaseState.FAILED) {
                        //todo: need access to https://tvdev.atlassian.net/wiki/spaces/TED/pages/591953929/CR+New+Purchase+on+screen+flow
                        when(purchaseResult.failReasonCode) {
                            20 -> {
                                //show Insufficient Funds dialog
                                binding.viewModel?.setSubscriptionBought(null)
                                showInsufficientFundsDialog(purchaseResult.failReasonCode.toString())

                            }
                            22 -> {
                                //show User does not exist dialog
                                binding.viewModel?.setSubscriptionBought(null)
                                showUserDoesNotExistsDialog(purchaseResult.failReasonCode.toString())
                            }
                            else -> {
                                //show generic error dialog? do nothing?
                                binding.viewModel?.setSubscriptionBought(null)
                                showPurchaseFailedDialog(purchaseResult.failReasonCode.toString())
                            }
                        }
                    }
                }
            })
    }

    override fun onDetach() {
        binding.viewModel?.hideOffers()
        super.onDetach()
    }

    override fun onDestroyView() {
        super.onDestroyView()
        nullableBinding = null
    }

    private fun showPurchaseSuccessDialog() {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_message),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_completed_close),
                MessageDialogCodes.purchaseSuccess
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showInsufficientFundsDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_insufficient_funds_close),
                MessageDialogCodes.insufficientFunds
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showPurchaseFailedDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_failed_close),
                MessageDialogCodes.purchaseFailed
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    private fun showUserDoesNotExistsDialog(code: String) {
        val messageDialogModel = MessageDialogModel(
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_title),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_message, code),
                requireContext().getString(com.twentyfouri.tvlauncher.common.R.string.purchase_user_does_not_exists_close),
                MessageDialogCodes.userDoesNotExists
        )
        val dialogFragment = MessageDialogFragment.newInstance(messageDialogModel)
        NavigatorCommon.getInstance().navigateDialog(dialogFragment)
    }

    companion object {
        private const val ARG_CHANNEL_ITEM = "ARG_CHANNEL_ITEM"

        fun newInstance(channelItem: SmartMediaItem): SubscriptionsFragment {
            val args = Bundle(1)
            args.putSerializable(ARG_CHANNEL_ITEM, channelItem)
            val fragment = SubscriptionsFragment()
            fragment.arguments = args
            return fragment
        }
    }

}