package com.twentyfouri.tvlauncher.ui

import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.fragment.app.commit
import androidx.lifecycle.Observer
import androidx.recyclerview.widget.RecyclerView
import com.google.firebase.analytics.FirebaseAnalytics
import com.google.firebase.crashlytics.FirebaseCrashlytics
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.adapters.ColumnItemAdapter
import com.twentyfouri.tvlauncher.data.ColumnItem
import com.twentyfouri.tvlauncher.data.ColumnItemType
import com.twentyfouri.tvlauncher.databinding.FragmentRecordingsBinding
import com.twentyfouri.tvlauncher.viewmodels.RecordingsViewModel
import com.twentyfouri.tvlauncher.widgets.RecordingItemView
import org.koin.androidx.viewmodel.ext.android.getViewModel
import timber.log.Timber

class RecordingsFragment: BaseFragment() {

    private var binding: FragmentRecordingsBinding? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View = FragmentRecordingsBinding.inflate(
        inflater,
        container,
        false
    ).apply {
        lifecycleOwner = this@RecordingsFragment.viewLifecycleOwner
        binding = this
    }.root

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        val viewModelFromFactory = getViewModel<RecordingsViewModel>()
        binding?.viewModel = viewModelFromFactory

        val columnItemAdapter = ColumnItemAdapter()
        binding?.recordingsVerticalRecycler?.adapter = columnItemAdapter

        setObserverOnColumnItems()

        binding?.recordingsVerticalRecycler?.requestFocus()
    }

    override fun onDestroyView() {
        binding = null
        super.onDestroyView()
    }

    private fun setObserverOnColumnItems() {
        binding?.recordingsVerticalRecycler?.addOnChildAttachStateChangeListener(object : RecyclerView.OnChildAttachStateChangeListener {
            override fun onChildViewAttachedToWindow(view: View) {
                val recordingItemViewModel = (view as? RecordingItemView)?.binding?.viewModel ?: return
                recordingItemViewModel.onItemClicked.observe(viewLifecycleOwner, Observer {
                   it?.let { onItemClicked(it) }
                })
            }

            override fun onChildViewDetachedFromWindow(view: View) {
                val recordingItemViewModel = (view as? RecordingItemView)?.binding?.viewModel ?: return
                recordingItemViewModel.onItemClicked.removeObservers(viewLifecycleOwner)
            }
        })
    }

    private fun setColumnItemsObserver() {
        binding?.viewModel?.columnItems?.observe(viewLifecycleOwner, Observer { columnItemList ->
            if (columnItemList.isNotEmpty()) {
                (binding?.recordingsVerticalRecycler?.adapter as? ColumnItemAdapter)?.apply {
                    submitList(columnItemList)
                    notifyDataSetChanged()
                }
            }
        })
    }

    override fun onStop() {
        binding?.viewModel?.recordingRefreshInProgress?.removeObservers(viewLifecycleOwner)

        super.onStop()
    }

    private fun onItemClicked(item: ColumnItem) {
        when(item.type) {
            ColumnItemType.RECORDING -> {
                if (binding?.viewModel?.recordingRefreshInProgress?.value == true) {
                    //wait for sync/refresh finish
                    val vm = binding?.viewModel as RecordingsViewModel
                    vm.showProgress()
                    vm.recordingRefreshInProgress.observe(viewLifecycleOwner, Observer {
                        if (it == false) {
                            continueToRecordingDetail(item)
                        }
                    })
                } else {
                    continueToRecordingDetail(item)
                }
            }
            ColumnItemType.RECORDING_EPISODES -> {
                val fragment = RecordingsEpisodesFragment()
                Timber.tag("RecordingsFragment.onItemClicked").d("commit $fragment")
                FirebaseCrashlytics.getInstance().log("RecordingsFragment.onItemClicked: commit $fragment")
                parentFragmentManager.commit {
                    replace(R.id.content_frame, fragment)
                    addToBackStack(null)
                }
                fragment.bind(item)
            }
            else -> {}
        }
    }

    private fun continueToRecordingDetail(item: ColumnItem) {
        binding?.viewModel?.columnItems?.removeObservers(viewLifecycleOwner)
        val intent = Intent(this.requireContext(), MainActivity::class.java)
        intent.action = MainActivity.INTENT_ACTION_DETAIL
        intent.putExtra(MainActivity.EXTRA_OPEN_USER_PROFILE, true)
        intent.putExtra("reference",item.recording?.reference)
        startActivity(intent)
    }

    override fun onPause() {
        super.onPause()
        binding?.viewModel?.columnItems?.removeObservers(viewLifecycleOwner)
    }

    override fun onResume() {
        super.onResume()
        FirebaseAnalytics.getInstance(requireContext()).logEvent(FirebaseAnalytics.Event.SCREEN_VIEW, bundleOf(
            FirebaseAnalytics.Param.SCREEN_NAME to "Recordings",
            FirebaseAnalytics.Param.SCREEN_CLASS to "RecordingsFragment"
        ))
        binding?.viewModel?.refreshRecordings()
        setColumnItemsObserver()
    }
}