package com.twentyfouri.tvlauncher.provider

import android.content.*
import android.database.Cursor
import android.net.Uri
import android.util.Log
import com.twentyfouri.tvlauncher.common.data.SetupData
import com.twentyfouri.tvlauncher.common.data.SetupDataDao
import com.twentyfouri.tvlauncher.common.data.SetupDataRepository
import com.twentyfouri.tvlauncher.BuildConfig
import timber.log.Timber

class CredentialsContentProvider: ContentProvider() {

    companion object {
        const val AUTHORITY = BuildConfig.APPLICATION_ID + ".provider"

        const val CODE_CREDENTIAL_DIR = 1
        const val CODE_CREDENTIAL_ITEM = 2

        private val MATCHER = UriMatcher(UriMatcher.NO_MATCH)

        init {
            MATCHER.addURI(AUTHORITY, SetupData.TABLE_NAME, CODE_CREDENTIAL_DIR)
            MATCHER.addURI(AUTHORITY, SetupData.TABLE_NAME + "/*", CODE_CREDENTIAL_ITEM)
        }
    }

    private lateinit var setupDataDao: SetupDataDao

    override fun query(
        uri: Uri,
        projection: Array<String>?,
        selection: String?,
        selectionArgs: Array<String>?,
        sortOrder: String?
    ): Cursor? {
        val code = MATCHER.match(uri)
        return when (code) {
            CODE_CREDENTIAL_DIR, CODE_CREDENTIAL_ITEM -> {
                val cursor = when (code) {
                    CODE_CREDENTIAL_DIR -> setupDataDao.selectAll()
                    CODE_CREDENTIAL_ITEM -> setupDataDao.selectById(ContentUris.parseId(uri))
                    else -> null
                }
                cursor?.setNotificationUri(context?.contentResolver, uri)
                return cursor
            }
            else -> throw IllegalArgumentException("Unknown URI: $uri")
        }
    }

    override fun getType(uri: Uri): String? =
        when (MATCHER.match(uri)) {
            CODE_CREDENTIAL_DIR -> "vnd.android.cursor.dir/${AUTHORITY}.${SetupData.TABLE_NAME}"
            CODE_CREDENTIAL_ITEM -> "vnd.android.cursor.item/${AUTHORITY}.${SetupData.TABLE_NAME}"
            else -> throw IllegalArgumentException("Unknown URI: $uri")
        }

    override fun insert(uri: Uri, values: ContentValues?): Uri? {
        Timber.tag("SessionStorage").d("CredentialsContentProvider insert")
        return when (MATCHER.match(uri)) {
            CODE_CREDENTIAL_DIR -> {
                val id = setupDataDao.insert(SetupData.fromContentValues(values!!))
                context!!.contentResolver.notifyChange(uri, null)
                ContentUris.withAppendedId(uri, id)
            }
            else -> throw java.lang.IllegalArgumentException("Unknown URI: $uri")
        }
    }

    override fun delete(uri: Uri, selection: String?, selectionArgs: Array<String>?): Int {
        return when (MATCHER.match(uri)) {
            CODE_CREDENTIAL_ITEM -> {
                val count = setupDataDao.deleteById(ContentUris.parseId(uri))
                context!!.contentResolver.notifyChange(uri, null)
                return count
            }
            else -> throw IllegalArgumentException("Unknown URI: $uri")
        }
    }

    override fun update(uri: Uri, values: ContentValues?, selection: String?, selectionArgs: Array<String>?): Int {
        return when (MATCHER.match(uri)) {
            CODE_CREDENTIAL_ITEM -> {
                val credentials = SetupData.fromContentValues(values!!)
                credentials.id = ContentUris.parseId(uri).toInt()

                val count = setupDataDao.update(credentials)
                context!!.contentResolver.notifyChange(uri, null)
                return count
            }
            else -> throw IllegalArgumentException("Unknown URI: $uri")
        }
    }

    override fun onCreate(): Boolean {
        setupDataDao = SetupDataRepository.getCredentialsDao(context!!)
        return true
    }
}