package com.twentyfouri.tvlauncher.images

import android.content.Context
import android.graphics.Bitmap
import android.net.Uri
import com.twentyfouri.smartmodel.model.dashboard.SmartMediaItem
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.provider.OpenFileProvider
import kotlinx.coroutines.*
import java.io.File
import java.io.FileOutputStream
import java.nio.file.Files
import java.util.*
import java.nio.file.attribute.BasicFileAttributeView

class SearchImageStorage(
    private val context: Context
) {
    private val renderer = SearchImageRenderer(context)
    private val resources = context.resources
    // Internal cache dirs are automatically cleaned up by Android when storage space is needed
    private val cachePath: File = File(context.cacheDir, context.getString(R.string.search_images_path))

    suspend fun getOrCreateChannelImage(channel: SmartMediaItem): Uri {
        val channelId = Flavor().getChannelId(channel)

        val file = getOrGenerate(channelId) {
            renderer.generateSearchChannelImage(channel)
        }

        return file?.let {
            OpenFileProvider.getUriForFile(context, context.getString(R.string.open_file_provider_authority), it)
        } ?: Uri.EMPTY
    }

    private suspend fun getOrGenerate(name: String, generateImage: suspend () -> Bitmap?): File? = withContext(Dispatchers.IO) {
        val fileName = sanitizeFilename(name)
        withContext(Dispatchers.IO) {
            cachePath.mkdirs()
            val file = File(cachePath, "$fileName.jpg")

            if (isFileStillFresh(file)) {
                file
            } else {
                generateImage()?.let { image ->
                    FileOutputStream(file).use {
                        image.compress(Bitmap.CompressFormat.JPEG, 100, it)
                    }
                }

                if (file.isFile) file else null
            }
        }
    }

    private fun isFileStillFresh(file: File): Boolean {
        val attr = try {
            Files.getFileAttributeView(file.toPath(), BasicFileAttributeView::class.java).readAttributes()
        } catch (e: java.nio.file.NoSuchFileException) {
            null
        }

        if (attr == null || !attr.isRegularFile) return false

        val now = System.currentTimeMillis()
        val lastModified = attr.lastModifiedTime().toMillis()
        val refreshInterval = resources.getInteger(R.integer.search_image_cache_refresh_interval)
        return (now - lastModified) < refreshInterval
    }

    private fun sanitizeFilename(name: String): String {
        return name.toLowerCase(Locale.US)
            .replace(" ", "-")
            .replace(Regex("[^A-Za-z0-9-]"), "")
    }
}