package com.twentyfouri.tvlauncher.homepagechannels

import android.app.job.JobInfo
import android.app.job.JobInfo.TriggerContentUri
import android.app.job.JobScheduler
import android.content.ComponentName
import android.content.ContentUris
import android.content.Context
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.graphics.Canvas
import android.graphics.drawable.VectorDrawable
import android.media.tv.TvContract
import android.net.Uri
import android.os.PersistableBundle
import android.util.Log
import androidx.tvprovider.media.tv.Channel
import androidx.tvprovider.media.tv.ChannelLogoUtils
import androidx.tvprovider.media.tv.TvContractCompat
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannel
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import timber.log.Timber

/** Manages interactions with the TV Provider. */
object TvUtil {

    private const val TAG = "TvUtil"
    private const val CHANNEL_JOB_ID_OFFSET: Long = 1000
    private val CHANNELS_PROJECTION = arrayOf(
        TvContractCompat.Channels._ID,
        TvContract.Channels.COLUMN_DISPLAY_NAME,
        TvContractCompat.Channels.COLUMN_BROWSABLE
    )

    /**
     * Converts a [Subscription] into a [Channel] and adds it to the tv provider.
     *
     * @param context used for accessing a content resolver.
     * @param homepageChannel to be converted to a channel and added to the tv provider.
     * @return the id of the channel that the tv provider returns.
     */
    suspend fun createChannel(
        context: Context,
        homepageChannel: HomepageChannel
    ): Long = withContext(Dispatchers.IO) outer@{ // Checks if our subscription has been added to the channels before.
        val cursor = context.contentResolver.query(
            TvContractCompat.Channels.CONTENT_URI,
            CHANNELS_PROJECTION,
            null,
            null,
            null
        )
        if (cursor != null && cursor.moveToFirst()) {
            do {
                val channel: Channel = Channel.fromCursor(cursor)
                if (homepageChannel.name.equals(channel.displayName)) {
                    Timber.d("Channel already exists. Returning channel ${channel.id} from TV Provider.")
                    return@outer channel.id
                }
            } while (cursor.moveToNext())
        }
        // Create the channel since it has not been added to the TV Provider.
        val appLinkIntentUri = Uri.parse(homepageChannel.appLinkIntentUri)
        val builder: Channel.Builder = Channel.Builder()
        builder.setType(TvContractCompat.Channels.TYPE_PREVIEW)
            .setDisplayName(homepageChannel.name)
            .setDescription(homepageChannel.description)
            .setAppLinkIntentUri(appLinkIntentUri)
            .setSystemChannelKey(homepageChannel.type.name) //Channel key used to identify channel for out-of-box configurations
        Timber.d("Creating channel: ${homepageChannel.name}")
        val channelUrl = context.contentResolver.insert(
            TvContractCompat.Channels.CONTENT_URI,
            builder.build().toContentValues()
        )
        channelUrl ?: return@outer 0L
        Timber.d("channel insert at $channelUrl")
        return@outer withContext(Dispatchers.Default) {
            val channelId = ContentUris.parseId(channelUrl)
            Timber.d("channel id $channelId")
            val bitmap: Bitmap = convertToBitmap(context, homepageChannel.channelLogo)
            ChannelLogoUtils.storeChannelLogo(context, channelId, bitmap)
            return@withContext channelId
        }
    }

    suspend fun getNumberOfChannels(context: Context): Int = withContext(Dispatchers.IO) {
        val cursor = context.contentResolver.query(
            TvContractCompat.Channels.CONTENT_URI,
            CHANNELS_PROJECTION,
            null,
            null,
            null
        )
        return@withContext cursor?.count ?: 0
    }

    /**
     * Converts a resource into a [Bitmap]. If the resource is a vector drawable, it will be
     * drawn into a new Bitmap. Otherwise the [BitmapFactory] will decode the resource.
     *
     * @param context used for getting the drawable from resources.
     * @param resourceId of the drawable.
     * @return a bitmap of the resource.
     */
    private suspend fun convertToBitmap(context: Context, resourceId: Int): Bitmap = withContext(Dispatchers.Default) {
        val drawable = context.getDrawable(resourceId)
        if (drawable is VectorDrawable) {
            val bitmap = Bitmap.createBitmap(
                drawable.getIntrinsicWidth(),
                drawable.getIntrinsicHeight(),
                Bitmap.Config.ARGB_8888
            )
            val canvas = Canvas(bitmap)
            drawable.setBounds(0, 0, canvas.width, canvas.height)
            drawable.draw(canvas)
            return@withContext bitmap
        }
        return@withContext BitmapFactory.decodeResource(context.resources, resourceId)
    }

    /**
     * Schedules syncing channels via a [JobScheduler].
     *
     * @param context for accessing the [JobScheduler].
     */
    fun scheduleSyncingChannel(context: Context) {
        val componentName = ComponentName(context, SyncChannelJobService::class.java)
        val builder = JobInfo.Builder(1, componentName)
        builder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY)
        val scheduler = context.getSystemService(Context.JOB_SCHEDULER_SERVICE) as JobScheduler
        Timber.d("Scheduled channel creation.")
        scheduler.schedule(builder.build())
    }

    /**
     * Schedulers syncing programs for a channel. The scheduler will listen to a [Uri] for a
     * particular channel.
     *
     * @param context for accessing the [JobScheduler].
     * @param channelId for the channel to listen for changes.
     */
    fun scheduleSyncingProgramsForChannel(context: Context, channelId: Long) {
        val componentName = ComponentName(context, SyncProgramsJobService::class.java)
        val builder = JobInfo.Builder(getJobIdForChannelId(channelId), componentName)
        val triggerContentUri = TriggerContentUri(
            TvContractCompat.buildChannelUri(channelId),
            TriggerContentUri.FLAG_NOTIFY_FOR_DESCENDANTS
        )
        builder.addTriggerContentUri(triggerContentUri)
        builder.setTriggerContentMaxDelay(0L)
        builder.setTriggerContentUpdateDelay(0L)
        val bundle = PersistableBundle()
        bundle.putLong(TvContractCompat.EXTRA_CHANNEL_ID, channelId)
        builder.setExtras(bundle)
        val scheduler = context.getSystemService(Context.JOB_SCHEDULER_SERVICE) as JobScheduler
        scheduler.cancel(getJobIdForChannelId(channelId))
        scheduler.schedule(builder.build())
    }

    private fun getJobIdForChannelId(channelId: Long): Int = (CHANNEL_JOB_ID_OFFSET + channelId).toInt()
}