package com.twentyfouri.tvlauncher.homepagechannels

import android.app.job.JobParameters
import android.app.job.JobService
import android.content.ContentUris
import android.content.Context
import android.net.Uri
import android.util.Log
import androidx.tvprovider.media.tv.Channel
import androidx.tvprovider.media.tv.PreviewProgram
import androidx.tvprovider.media.tv.TvContractCompat
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannelItem
import kotlinx.coroutines.*
import org.koin.core.component.KoinComponent
import org.koin.core.component.get
import timber.log.Timber
import java.util.*

/**
 * Syncs programs for a channel. A channel id is required to be passed via the [ ].
 * This service is scheduled to listen to changes to a channel. Once the job
 * completes, it will reschedule itself to listen for the next change to the channel.
 * See [ ][TvUtil.scheduleSyncingProgramsForChannel] for more details about the scheduling.
 */
class SyncProgramsJobService : JobService(), KoinComponent {

    companion object {
        private const val TAG = "SyncProgramsJobService"
        private var syncJob: Job? = null
    }

    override fun onStartJob(jobParameters: JobParameters): Boolean {
        Timber.d("onStartJob(): $jobParameters")
        val channelId = getChannelId(jobParameters)
        if (channelId == -1L) {
            return false
        }
        Timber.d("onStartJob(): Scheduling syncing for programs for channel $channelId")
        syncJob = CoroutineScope(Dispatchers.Main).launch {
            syncChannelItems(applicationContext, channelId)
            //Daisy chain listening for the next change to the channel.
            TvUtil.scheduleSyncingProgramsForChannel(this@SyncProgramsJobService, channelId)
            syncJob = null
            jobFinished(jobParameters, false)
        }
        return true
    }

    override fun onStopJob(jobParameters: JobParameters): Boolean {
        try {
            syncJob?.cancel()
        } catch ( e: CancellationException ) { }
        return true
    }

    private fun getChannelId(jobParameters: JobParameters): Long {
        val extras = jobParameters.extras ?: return -1L
        return extras.getLong(TvContractCompat.EXTRA_CHANNEL_ID, -1L)
    }

    /*
     * Syncs programs by querying the given channel id.
     *
     * If the channel is not browsable, the programs will be removed to avoid showing
     * stale programs when the channel becomes browsable in the future.
     *
     * If the channel is browsable, then it will check if the channel has any programs.
     *      If the channel does not have any programs, new programs will be added.
     *      If the channel does have programs, then a fresh list of programs will be fetched and the
     *          channel's programs will be updated.
     */
    private suspend fun syncChannelItems(channelId: Long, initialMovies: List<HomepageChannelItem>) = withContext(Dispatchers.IO) {
        Timber.d("Sync programs for channel: $channelId")
        var channelItems: List<HomepageChannelItem> = initialMovies
        contentResolver
            .query(
                TvContractCompat.buildChannelUri(channelId),
                null,
                null,
                null,
                null
            ).use { cursor ->
                if (cursor != null && cursor.moveToNext()) {
                    val channel: Channel = Channel.fromCursor(cursor)
                    if (!channel.isBrowsable) {
                        Timber.d("Channel is not browsable: $channelId")
                        deleteChannelItems(channelId, channelItems)
                    } else {
                        Timber.d("Channel is browsable: $channelId")
                        channelItems = if (channelItems.isEmpty()) {
                            createChannelItems(channelId,
                                get<HomepageChannelNetworkRepository>().getList(applicationContext,
                                    channelId))
                        } else {
                            updateChannelItems(channelId, channelItems)
                        }
                        HomepageChannelDatabaseRepository.saveChannelItems(applicationContext,
                            channelId,
                            channelItems)
                    }
                }
            }
    }

    private suspend fun createChannelItems(channelId: Long, items: List<HomepageChannelItem>): List<HomepageChannelItem> = withContext(Dispatchers.IO) {
        val channelItemsAdded: MutableList<HomepageChannelItem> = ArrayList(items.size)
        for (movie in items) {
            val previewProgram: PreviewProgram = buildChannelItem(channelId, movie)
            val programUri = contentResolver
                .insert(
                    TvContractCompat.PreviewPrograms.CONTENT_URI,
                    previewProgram.toContentValues()
                )
            programUri ?: continue
            val programId = ContentUris.parseId(programUri)
            Timber.d("Inserted new program: $programId")
            movie.programId = programId
            channelItemsAdded.add(movie)
        }
        return@withContext channelItemsAdded
    }

    private suspend fun updateChannelItems(
        channelId: Long,
        items: List<HomepageChannelItem?>
    ): List<HomepageChannelItem> = withContext(Dispatchers.IO) { // By getting a fresh list, we should see a visible change in the home screen.
        val updateChannelItems: List<HomepageChannelItem> = get<HomepageChannelNetworkRepository>().getList(applicationContext, channelId)
        for (i in items.indices) {
            val old: HomepageChannelItem? = items[i]
            val update: HomepageChannelItem? = updateChannelItems[i]
            val programId: Long = old?.programId ?: 0
            contentResolver
                .update(
                    TvContractCompat.buildPreviewProgramUri(programId),
                    buildChannelItem(channelId, update).toContentValues(),
                    null,
                    null
                )
            Timber.d("Updated program: $programId")
            update?.programId = programId
        }
        return@withContext updateChannelItems
    }

    private suspend fun deleteChannelItems(channelId: Long, items: List<HomepageChannelItem?>) = withContext(Dispatchers.IO) {
        if (items.isEmpty()) return@withContext
        var count = 0
        for (movie in items) {
            count += contentResolver
                .delete(
                    TvContractCompat.buildPreviewProgramUri(movie?.programId ?: 0),
                    null,
                    null
                )
        }
        Timber.d("Deleted $count items for  channel $channelId")
        // Remove our local records to stay in sync with the TV Provider.
        HomepageChannelDatabaseRepository.removeChannelItems(applicationContext, channelId)
    }

    private suspend fun buildChannelItem(channelId: Long, channelItem: HomepageChannelItem?): PreviewProgram = withContext(Dispatchers.Default) {
        val posterArtUri = Uri.parse(channelItem?.cardImageUrl ?: "")
//        val appLinkUri: Uri? = AppLinkHelper.buildPlaybackUri(channelId, movie?.id ?: 0)
        val appLinkUri: Uri? = AppLinkHelper.buildPlaybackUri(
            channelId = channelId,
            channelItemId = channelItem?.id ?: -1,
            position = AppLinkHelper.DEFAULT_POSITION.toLong(),
            scheme = getString(R.string.app_link_activity_scheme),
            host = getString(R.string.app_link_activity_host),
            appLinkJson = channelItem?.appLinkUri ?: channelItem?.id.toString()
        )
        val previewVideoUri = Uri.parse(channelItem?.videoUrl ?: "")
        val iconUrl = Uri.parse(channelItem?.iconUrl ?: "")
        val builder: PreviewProgram.Builder = PreviewProgram.Builder()
        builder
            .setChannelId(channelId)
            .setType(TvContractCompat.PreviewProgramColumns.TYPE_TV_EPISODE)
            .setTitle(channelItem?.title)
            .setDescription(channelItem?.description)
            .setGenre(channelItem?.category)
            .setReleaseDate(channelItem?.date)
            .setPosterArtUri(posterArtUri)
            .setLogoUri(iconUrl)
            .setPreviewVideoUri(previewVideoUri)
            .setIntentUri(appLinkUri)
        return@withContext builder.build()
    }

    private suspend fun syncChannelItems(context: Context, channelId: Long) = withContext(Dispatchers.Default) {
        HomepageChannelDatabaseRepository
            .findHomepageChannelByChannelId(context, channelId)
            ?.let { HomepageChannelDatabaseRepository.getChannelItems(context, channelId) }
            ?.also { cachedHomepageChannelItems -> syncChannelItems(channelId, cachedHomepageChannelItems) }
    }
}