package com.twentyfouri.tvlauncher.homepagechannels

import android.content.Context
import android.util.Log
import com.twentyfouri.smartmodel.FlowSmartApi
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.media.SmartMediaStreamOptions
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.ImageType
import com.twentyfouri.tvlauncher.R
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannel
import com.twentyfouri.tvlauncher.homepagechannels.room.HomepageChannelItem
import kotlinx.coroutines.*
import kotlinx.coroutines.flow.collect
import kotlinx.coroutines.flow.first
import org.joda.time.format.DateTimeFormat
import timber.log.Timber

/** gathering HomepageChannels from an external source.  */
class HomepageChannelNetworkRepository(
    private val smartApi: FlowSmartApi?
) {

    private var count: Long = 0

    /**
     * Creates a list of home page channels that every users should have.
     *
     * @param context used for accessing shared preferences.
     * @return a list of default homepage channels.
     */
    suspend fun createHomepageChannels(context: Context): List<HomepageChannel> = withContext(Dispatchers.IO) {
        val appName = context.getString(R.string.app_name)
        val allChannelsChannel = HomepageChannel(
            name = context.getString(R.string.homepage_channel_all_channels_title),
            description = context.getString(R.string.homepage_channel_all_channels_title),
            appLinkIntentUri = AppLinkHelper.buildBrowseUri(
                channelName = appName,
                defaultNavigationTarget = AppLinkActivity.EPG,
                scheme = context.getString(R.string.app_link_activity_scheme),
                host = context.getString(R.string.app_link_activity_host)
            ).toString(),
            channelLogo = R.drawable.ic_oval,
            type = HomepageChannel.Type.ALL_CHANNELS
        )
        val onNowChannel = HomepageChannel(
            name = context.getString(R.string.homepage_channel_on_now_title),
            description = context.getString(R.string.homepage_channel_on_now_title),
            appLinkIntentUri = AppLinkHelper.buildBrowseUri(
                channelName = appName,
                defaultNavigationTarget = AppLinkActivity.HOME,
                scheme = context.getString(R.string.app_link_activity_scheme),
                host = context.getString(R.string.app_link_activity_host)
            ).toString(),
            channelLogo = R.drawable.ic_oval,
            type = HomepageChannel.Type.ON_NOW
        )
        return@withContext listOf(allChannelsChannel, onNowChannel)
    }

    /**
     * Creates and caches a list of channel items.
     *
     * @return a list of channel items.
     */
    suspend fun getList(context: Context, channelId: Long): List<HomepageChannelItem> = withContext(Dispatchers.Default){
        val homepageChannel: HomepageChannel? = HomepageChannelDatabaseRepository.findHomepageChannelByChannelId(context, channelId)
        return@withContext when (homepageChannel?.type) {
            HomepageChannel.Type.ALL_CHANNELS -> getAllChannelsList(context)
            HomepageChannel.Type.ON_NOW -> getOnNowList(context)
            else -> createMovieList()
        }
    }

    private suspend fun getAllChannelsList(context: Context): List<HomepageChannelItem> = withContext(Dispatchers.Default){
        val pageSection = Flavor().getHomepageChannel_SmartPageSection_AllChannels() ?: return@withContext emptyList<HomepageChannelItem>()
        return@withContext smartApi?.getHomepageChannelContent(pageSection) ?: emptyList()
    }

    private suspend fun getOnNowList(context: Context): List<HomepageChannelItem> = withContext(Dispatchers.Default) {
        val pageSection = Flavor().getHomepageChannel_SmartPageSection_OnNow() ?: return@withContext emptyList<HomepageChannelItem>()
        return@withContext smartApi?.getHomepageChannelContent(pageSection) ?: emptyList()
    }

    private suspend fun FlowSmartApi.getHomepageChannelContent(section: SmartPageSection): List<HomepageChannelItem> = coroutineScope {
        var channelItems: List<HomepageChannelItem> = emptyList()

        val sectionContent: List<SmartMediaItem> = getSectionContent(section)
        channelItems = sectionContent.mapIndexed { index, smartMediaItem ->
            val detail = withContext(Dispatchers.IO) {
                Timber.tag("MediaDetail").d("getMediaDetail launcher from getHomepageChannelContent")
                getMediaDetail(smartMediaItem.reference) }.first()
            val url = try {
                detail
                    .editions.firstOrNull()
                    ?.streams?.firstOrNull()
                    ?.let { withContext(Dispatchers.IO) { getMediaStream(it, SmartMediaStreamOptions(SmartMediaStreamOptions.Builder())).first() } }
                    ?.primaryUrl
            } catch (e: Exception) {
                null
            }
            val imageUrl = Flavor().pickBasedOnFlavor(Flavor().getImageOfType(smartMediaItem, ImageType.LIGHT),1, 1, SmartImages.UNRESTRICTED)
            when (section.sectionStyle) {
                Flavor().getAllChannelsSectionStyle() -> {
                    HomepageChannelItem(
                        id = index.toLong(),
                        title = smartMediaItem.title,
                        description = smartMediaItem.description,
                        studio = "default studio",
                        cardImageUrl = imageUrl,
                        backgroundImageUrl = imageUrl,
                        videoUrl = url ?: "",
                        appLinkUri = Flavor().getSmartMediaUriParamsFromReference(smartMediaItem.reference)?.toString()
                    )
                }
                else -> {
                    val logo = Flavor(). pickBasedOnFlavor(Flavor().getImageOfType(smartMediaItem, ImageType.OVERLAY),1, 1, SmartImages.UNRESTRICTED)
                    val formatter = DateTimeFormat.forPattern("yyyy-MM-dd")
                    HomepageChannelItem(
                        id = index.toLong(),
                        title = smartMediaItem.title,
                        description = smartMediaItem.description,
                        date = formatter.print(detail.startDate),
                        studio = "default studio",
                        category = smartMediaItem.categories.firstOrNull()?.label,
                        cardImageUrl = imageUrl,
                        backgroundImageUrl = imageUrl,
                        iconUrl = logo,
                        videoUrl = url ?: "",
                        appLinkUri = Flavor().getSmartMediaUriParamsFromReference(smartMediaItem.reference)?.toString()
                    )
                }
            }
        }
        return@coroutineScope channelItems
    }

    private suspend fun FlowSmartApi.getSectionContent(section: SmartPageSection): List<SmartMediaItem> = withContext(Dispatchers.Default) {
        val playlistSelectedOptions = getPlaylistOptions(section.playlistReference).first().apply {
            pagings.setupFixedMode(50)
            sortings.add(
                SmartPlaylistSorting(
                    reference = object: SmartPlaylistSortingReference() {},
                    label = "empty"
                )
            )
        }.createSelection()

        lateinit var playlist: SmartPlaylist
        getPlaylistItems(section.playlistReference, playlistSelectedOptions).collect { playlist = it }
        return@withContext playlist.items
    }

    private fun createMovieList(): List<HomepageChannelItem> {
        val list: MutableList<HomepageChannelItem> = ArrayList()
        val title = arrayOf(
            "Zeitgeist 2010_ Year in Review",
            "Google Demo Slam_ 20ft Search",
            "Introducing Gmail Blue",
            "Introducing Google Fiber to the Pole",
            "Introducing Google Nose"
        )
        val description = ("Fusce id nisi turpis. Praesent viverra bibendum semper. "
                + "Donec tristique, orci sed semper lacinia, quam erat rhoncus massa, non congue tellus est "
                + "quis tellus. Sed mollis orci venenatis quam scelerisque accumsan. Curabitur a massa sit "
                + "amet mi accumsan mollis sed et magna. Vivamus sed aliquam risus. Nulla eget dolor in elit "
                + "facilisis mattis. Ut aliquet luctus lacus. Phasellus nec commodo erat. Praesent tempus id "
                + "lectus ac scelerisque. Maecenas pretium cursus lectus id volutpat.")
        val studio = arrayOf(
            "Studio Zero", "Studio One", "Studio Two", "Studio Three", "Studio Four"
        )
        val videoUrl = arrayOf(
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Zeitgeist/Zeitgeist%202010_%20Year%20in%20Review.mp4",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Demo%20Slam/Google%20Demo%20Slam_%2020ft%20Search.mp4",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Gmail%20Blue.mp4",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Fiber%20to%20the%20Pole.mp4",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Nose.mp4"
        )
        val bgImageUrl = arrayOf(
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Zeitgeist/Zeitgeist%202010_%20Year%20in%20Review/bg.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Demo%20Slam/Google%20Demo%20Slam_%2020ft%20Search/bg.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Gmail%20Blue/bg.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Fiber%20to%20the%20Pole/bg.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Nose/bg.jpg"
        )
        val cardImageUrl = arrayOf(
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Zeitgeist/Zeitgeist%202010_%20Year%20in%20Review/card.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/Demo%20Slam/Google%20Demo%20Slam_%2020ft%20Search/card.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Gmail%20Blue/card.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Fiber%20to%20the%20Pole/card.jpg",
            "http://commondatastorage.googleapis.com/android-tv/Sample%20videos/April%20Fool's%202013/Introducing%20Google%20Nose/card.jpg"
        )
        for (index in title.indices) {
            list.add(
                buildMovieInfo(
                    "category",
                    title[index],
                    description,
                    studio[index],
                    videoUrl[index],
                    cardImageUrl[index],
                    bgImageUrl[index]
                )
            )
        }
        return list
    }

    private fun buildMovieInfo(
        category: String,
        title: String,
        description: String,
        studio: String,
        videoUrl: String,
        cardImageUrl: String,
        backgroundImageUrl: String
    ): HomepageChannelItem {
        val movie = HomepageChannelItem(
            id = count,
            title = title,
            description = description,
            studio = studio,
            category = category,
            cardImageUrl = cardImageUrl,
            backgroundImageUrl = backgroundImageUrl,
            videoUrl = videoUrl
        )
        count++
        return movie
    }
}
