package com.twentyfouri.tvlauncher.di

import com.twentyfouri.smartmodel.epg.EpgDatabase
import com.twentyfouri.tvlauncher.Flavor
import com.twentyfouri.tvlauncher.common.analytics.YouboraAnalytics
import com.twentyfouri.tvlauncher.common.data.*
import com.twentyfouri.tvlauncher.common.data.apihandler.ApiHandler
import com.twentyfouri.tvlauncher.common.utils.NavigatorCommon
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLogger
import com.twentyfouri.tvlauncher.common.utils.logging.OselToggleableLoggerPermissions
import com.twentyfouri.tvlauncher.data.*
import com.twentyfouri.tvlauncher.homepagechannels.HomepageChannelNetworkRepository
import com.twentyfouri.tvlauncher.notifications.NotificationsSidePanelActivity
import com.twentyfouri.tvlauncher.notifications.ReadNotifications
import com.twentyfouri.tvlauncher.ui.*
import com.twentyfouri.tvlauncher.utils.Navigator
import com.twentyfouri.tvlauncher.viewmodels.*
import org.koin.androidx.viewmodel.dsl.viewModel
import org.koin.core.qualifier.named
import org.koin.dsl.bind
import org.koin.dsl.module

@JvmField
val persistedModule = module {
    single(createdAtStart = true) { OselToggleableLoggerPermissions(context = get()) }
    single { OselToggleableLogger(permissions = get()) }
}

@JvmField
val repositoryModule = module {
    single { Navigator.getInstance() } bind NavigatorCommon::class
    single { ResourceRepository(context = get()) }
    single { AppUsageRepository(AppUsageDatabase.getInstance(context = get()).appUsageDao()) }
    single(named("credentialsRepository")) { SetupDataRepository.getInstance(context = get()) }
    single {
        try {
            SmartApiRepository.getSmartApi(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                oselLogger = get()
            )
        } catch (exception: Exception) {
            SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                oselLogger = get()
            ) ?: throw ApiMissingException()
        }
    }
    single { AppListDao(c = get(), AppUsageDatabase.getInstance(context = get())) }
    single { AppListRepository(appListDao = get())}
    single {
        ContentResolverRepository(
            context = get(),
            readNotifications = get()
        )
    }
    single {
        ApiHandler(
            resourceRepository = get(),
            context = get(),
            navigatorCommon = get()
        )
    }
    single {
        RecordingSettingsRepository(
            context = get()
        )
    }
    single {
        EpgRepository(
            smartApi = get(),
            apiHandler = get(),
            resourceRepository = get()
        )
    }
    single {
        RecordingsRepository(
            smartApi = get(),
            apiHandler = get(),
            recordingsSettingsRepository = get(),
            epgRepository = get()
        )
    }
    single {
        RowPageRepository(
            smartApi = SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                get()
            ),
            appListRepository = get(),
            contentResolverRepository = get(),
            apiHandler = get(),
            context = get()
        )
    }
    single { DateTimeRepository(c = get()) }
    single {
        MetadataRepository(
            smartApi = SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                get()
            ),
            apiHandler = get(),
            resourceRepository = get(),
            youboraAnalytics = get(),
            epgDao = EpgDatabase.getInstance(get()).epgDao
        )
    }
    single { YouboraAnalytics.getInstance(context = get()) }
    single {
        MenuRepository(
            smartApi = SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                get()
            ),
            apiHandler = get()
        )
    }
    single {
        PersonalSettingsRepository(
            setupDataRepository = get(named("credentialsRepository")),
            smartApi = get(),
            apiHandler = get()
        )
    }
    single {
        HomepageChannelNetworkRepository(
            smartApi = SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                get()
            )
        )
    }
    factory {
        RecommendationsRepository(
            smartApi = get(),
            apiHandler = get()
        )
    }
    single { ReadNotifications(context = get()) }
}

@JvmField
val viewModelModule = module {
    viewModel {
        RowItemViewModel(
            navigator = get(),
            resourceRepository = get(),
            appUsageRepository = get()
        )
    }

    scope<RowPageFragment> {
        viewModel {
            RowPageViewModel(
                rowPageRepository = get(),
                setupDataRepository = get(named("credentialsRepository")),
                recordingsRepository = try {
                    get<RecordingsRepository>()
                } catch (t: ApiMissingException) {
                    null
                },
                resourceRepository = get(),
                epgRepository = get()
            )
        }
    }

    scope<EpgFragment> {
        viewModel {
            EpgViewModel(
                repository = get(),
                recordingsRepository = get(),
                resourceRepository = get(),
                context = get()
            )
        }
    }
    scope<DetailFragment> {
        viewModel {
            DetailViewModel(
                navigator = get(),
                dateTimeRepository = get(),
                epgRepository = try {
                    get<EpgRepository>()
                } catch (t: ApiMissingException) {
                    null
                },
                resourceRepository = get(),
                apiHandler = get(),
                rowPageRepository = get()
            )
        }
    }

    scope<ListPickerFragment> {
        viewModel { parameters ->
            ListPickerViewModel(title = parameters.get())
        }
    }

    scope<FavoriteAppsActivity> {
        viewModel{
            FavoriteAppsViewModel(
                appListRepository = get()
            )
        }
    }

    scope<FavoriteAppPickerFragment>{
        viewModel {
            FavoriteAppsPickerViewModel(
                appListRepository = get()
            )
        }
    }

    scope<RecordingsEpisodesFragment> {
        viewModel {
            RecordingsEpisodesViewModel(
                resourceRepository = get(),
                recordingsRepository = get(),
                navigator = get()
            )
        }
    }

    scope<RecordingsFragment> {
        viewModel {
            RecordingsViewModel(
                resourceRepository = get(),
                recordingsRepository = get(),
                navigator = get()
            )
        }
    }

    scope<SubscriptionsFragment> {
        viewModel {
            SubscriptionsViewModel(
                metadataRepository = get(),
                resourceRepository = get()
            )
        }
    }

    viewModel {
        OfflineRowViewModel(
            navigator = get(),
            resourceRepository = get()
        )
    }

    scope<SubscriptionsGridFragment> {
        viewModel {
            SubscriptionsGridViewModel(
                metadataRepository = get(),
                resourceRepository = get()
            )
        }
    }

    viewModel {
        MetadataViewModel(
            repository = get(),
            dateTimeRepository = get(),
            recordingsRepository = try {
                get<RecordingsRepository>()
            } catch (t: ApiMissingException) {
                null
            },
            resourceRepository = get()
        )
    }

    scope<PlayerFragment> {
        viewModel {
            PlayerViewModel(
                epgRepository = get(),
                metadataRepository = get(),
                dateTimeRepository = get()
            )
        }
        viewModel {
            PlayerUICatchupViewModel(
                epgRepository = get(),
                dateTimeRepository = get(),
                metadataRepository = get(),
                resourceRepository = get(),
                youboraAnalytics = get()
            )
        }
        viewModel {
            PlayerUILiveViewModel(
                epgRepository = get(),
                metadataRepository = get(),
                dateTimeRepository = get(),
                recordingsRepository = get(),
                resourceRepository = get(),
                youboraAnalytics = get()
            )
        }
        viewModel {
            PlayerUINoControlsViewModel(
                epgRepository = get(),
                metadataRepository = get(),
                dateTimeRepository = get(),
                resourceRepository = get(),
                youboraAnalytics = get()
            )
        }
        viewModel {
            PlayerUIVodViewModel(
                dateTimeRepository = get(),
                metadataRepository = get(),
                resourceRepository = get(),
                youboraAnalytics = get()
            )
        }
    }

    // this is intentionally not defined in scope of MainActivity as we need to have the viewModel well before onCreateView because
    // of serial number checking and connectivity checking functionality
    viewModel<MainActivityViewModel> {
        MainActivityViewModel(
            setupDataRepository = get(named("credentialsRepository")),
            menuRepository = get(),
            epgRepository = get(),
            smartModelApi = SmartApiRepository.getSmartApi_safe(
                context = get(),
                setupDataRepo = get(named("credentialsRepository")),
                get()
            ),
            apiHandler = get(),
            youboraAnalytics = get()
        )
    }

    scope<MainActivity> {
        // do not use viewModel Koin provider as due to serial number checks and network checks we need to have the viewModel
        // before onCreateView happens, which is otherwise the point when the viewModel provider starts to work
    }

    scope<DiagnosticsActivity> {}

    scope<DiagnosticsNetworkActivity> {}

    scope<DiagnosticsSystemActivity> {}

    scope<NotificationsSidePanelActivity>{}

    scope<PersonalSettingsActivity> {
        viewModel {
            PersonalSettingsViewModel(
                personalSettingsRepository = get(),
                recordingsRepository = get()
            )
        }
    }

    viewModel {
        RecordingItemViewModel(
            resourceRepository = get(),
            dateTimeRepository = get()
        )
    }

    viewModel { FloatingLabelViewModel() }

    viewModel { Flavor().getMediaRestrictionViewModel() }

    viewModel {
        MetadataAppChannelProgramViewModel(
            dateTimeRepository = get(),
            resourceRepository = get()
        )
    }

    scope<MockFragment> {
        viewModel { MockViewModel() }
    }

    scope<TopbarFragment> {
        viewModel {
            TopbarViewModel(
                menuRepository = get(),
                contentResolverRepository = get(),
                resourceRepository = get()
            )
        }
    }
}

val allModules = persistedModule + repositoryModule + viewModelModule