package com.twentyfouri.tvlauncher

import android.net.Uri
import android.view.KeyEvent
import android.view.KeyEvent.*
import com.google.gson.Gson
import com.google.gson.JsonElement
import com.twentyfouri.androidcore.utils.EmptyImageSpecification
import com.twentyfouri.smartmodel.eventsOnChannel
import com.twentyfouri.smartmodel.backstage.BackstageReferenceFactory
import com.twentyfouri.smartmodel.backstage.reference.BackstageMediaReference
import com.twentyfouri.smartmodel.backstage.reference.BackstagePageReference
import com.twentyfouri.smartmodel.backstage.reference.BackstagePlaylistReference
import com.twentyfouri.smartmodel.backstage.reference.BackstageStreamReference
import com.twentyfouri.smartmodel.model.dashboard.*
import com.twentyfouri.smartmodel.model.media.SmartMediaDetail
import com.twentyfouri.smartmodel.model.menu.SmartMenu
import com.twentyfouri.smartmodel.model.menu.SmartMenuItem
import com.twentyfouri.smartmodel.model.payment.SmartPurchase
import com.twentyfouri.smartmodel.model.payment.SmartPurchaseState
import com.twentyfouri.smartmodel.model.payment.SmartSubscription
import com.twentyfouri.smartmodel.model.payment.SmartSubscriptionReference
import com.twentyfouri.smartmodel.model.search.SmartGenreFilterReference
import com.twentyfouri.smartmodel.model.user.SmartSessionState
import com.twentyfouri.smartmodel.serialization.SmartDataValue
import com.twentyfouri.tvlauncher.data.*
import com.twentyfouri.tvlauncher.ui.EpgFragment
import com.twentyfouri.tvlauncher.utils.KeyEventAction

private const val TIMEZONE_ID = "Europe/Amsterdam"

class Flavor : FlavorBase() {

    override fun shouldHandleGuideButton(): Boolean = true

    override val getEpgConfigValues: EpgFragment.EPGConfigValues = EPGConfigValuesBackstage()

    override fun compareRecordingWithEvent(
        recording: SmartMediaItem?,
        event: SmartMediaItem?
    ): Boolean = false

    override fun allowDisplayOfExternalRows() = BuildConfig.DEBUG

    override suspend fun isLoggedIn(smartApi: eventsOnChannel): Boolean {
        //TODO
        return true
//        return when (smartApi.restoreSessionImmediate()) {
//            SmartSessionState.LOGGED_IN -> true
//            else -> false
//        }
    }

    override fun getPageReference(type: PageType, menus: List<SmartMenu>?): SmartPageReference {
        //TODO
        return when (type) {
            PageType.MAIN,
            PageType.EPG -> menus?.firstOrNull()?.items?.firstOrNull()?.target?.pageReference
                    ?: HardcodedPageReference(HardcodedPageReference.Type.APPS)
            PageType.APPS -> HardcodedPageReference(HardcodedPageReference.Type.APPS)
        }
    }

    override fun getPlaylistReference(type: PlaylistType) = getPlaylistReference(type, null)

    override fun getPlaylistReference(
        type: PlaylistType,
        externalReference: SmartMediaReference?
    ): SmartPlaylistReference {
        //TODO
        return when (type) {
            PlaylistType.ALL_CHANNELS -> BackstagePlaylistReference.forChannels()
            else -> throw IllegalArgumentException("Incorrect playlist type")
        }
    }

    override fun getGenreFilterReference(title: String, type: PlaylistType?) = BackstageGenreFilterReference(title)

    class BackstageGenreFilterReference(
            override val title: String
    ): SmartGenreFilterReference() {
        override fun equals(other: Any?): Boolean = true
        override fun hashCode(): Int = title.hashCode()
    }

    override fun convertMediaReferenceToDetail(reference: SmartMediaReference) = SmartMediaDetail(reference, SmartMediaType.UNKNOWN)

    override fun getChannelId(mediaItem: SmartMediaItem) = (mediaItem.reference as BackstageMediaReference).id

    override fun getImageOfType(mediaItem: SmartMediaItem, imageType: ImageType): SmartImages {
        return when(imageType) {
            ImageType.DARK -> {
                mediaItem.images
            }
            ImageType.LIGHT -> {
               mediaItem.images
            }
            ImageType.OVERLAY -> {
                SmartImages()
            }
        }
    }

    override fun getMenuType(menuType: String) = when (menuType) {
        HardcodedMenuMapper.Type.SEARCH.name -> MenuType.SEARCH
        HardcodedMenuMapper.Type.NOTIFICATIONS.name -> MenuType.NOTIFICATIONS
        HardcodedMenuMapper.Type.VOICE.name -> MenuType.VOICE
        HardcodedMenuMapper.Type.SETTINGS.name -> MenuType.SETTINGS
        HardcodedMenuMapper.Type.APPS.name -> MenuType.APPS
        HardcodedMenuMapper.Type.PERSONAL.name -> MenuType.PERSONAL
        else -> MenuType.UNKNOWN
    }

    override fun getCatchupMediaStreamReference(
        detail: SmartMediaDetail,
        channel: SmartMediaItem?
    ): BackstageStreamReference? = detail.editions.firstOrNull()?.streams?.firstOrNull() as BackstageStreamReference

    override fun getSmartMediaUriParamsFromReference(reference: SmartMediaReference): Uri? =
            Uri.parse(reference.toSmartDataValue().toJsonTree().toString())

    override fun getSmartMediaReferenceFromUri(uri: Uri): SmartMediaReference? =
            BackstageReferenceFactory().toSmartMediaReference(SmartDataValue.fromJson(Gson().fromJson(uri.toString(), JsonElement::class.java)))

    override suspend fun getLauncherPage(
        smartApi: eventsOnChannel?,
        pageReference: SmartPageReference
    ): SmartPage? {
        val page = smartApi?.getPage(pageReference)
        val mainPageId = "todo" //TODO
        if (pageReference is BackstagePageReference) {
            if (pageReference.pageId == mainPageId) {
                    // insert recommended apps
                val sections = page?.sections?.toMutableList() ?: emptyList<SmartPageSection>().toMutableList()
                sections.add(0, HardcodedPageMapper.convertRecommendedApps()) // Recommended Apps row positioned first in case On Now is missing
                page?.sections = sections//.take(1)
            }
        }
        return page
    }

    override suspend fun getSubscriptions(
        smartApi: eventsOnChannel,
        reference: SmartMediaReference
    ) = emptyList<SmartSubscription>()

    override suspend fun subscribeProduct(
        smartApi: eventsOnChannel,
        productReference: SmartSubscriptionReference?,
        isAutoRenew: Boolean?,
        subscriptionProduct: SmartSubscription?
    ) : SmartPurchase = SmartPurchase(-1, SmartPurchaseState.FAILED)

    override fun getTimezoneID(): String? = TIMEZONE_ID

    override fun getActionForKeyEvent(event: KeyEvent): KeyEventAction {
        return when (event.keyCode){
            KEYCODE_BACK,
            KEYCODE_BUTTON_B -> KeyEventAction.BACK
            KEYCODE_MEDIA_STOP -> KeyEventAction.STOP
            KEYCODE_MEDIA_PLAY_PAUSE -> KeyEventAction.PLAY_PAUSE
            KEYCODE_MEDIA_REWIND -> KeyEventAction.RW
            KEYCODE_MEDIA_FAST_FORWARD -> KeyEventAction.FF
            KEYCODE_CHANNEL_DOWN -> KeyEventAction.CHANNEL_DOWN
            KEYCODE_CHANNEL_UP -> KeyEventAction.CHANNEL_UP
            KEYCODE_DPAD_LEFT -> KeyEventAction.DPAD_LEFT
            KEYCODE_DPAD_RIGHT -> KeyEventAction.DPAD_RIGHT
            KEYCODE_DPAD_DOWN -> KeyEventAction.DPAD_DOWN
            KEYCODE_DPAD_UP -> KeyEventAction.DPAD_UP
            KEYCODE_DPAD_CENTER -> KeyEventAction.DPAD_CENTER
            in KEYCODE_NUMPAD_0..KEYCODE_NUMPAD_9,
            in KEYCODE_0..KEYCODE_9 -> KeyEventAction.NUMBER
            KEYCODE_LAST_CHANNEL -> KeyEventAction.LAST
            KEYCODE_PROG_GREEN -> KeyEventAction.STARTOVER
            KEYCODE_INFO -> KeyEventAction.INFO
            KEYCODE_MEDIA_RECORD -> KeyEventAction.RECORD
            else -> return KeyEventAction.NO_ACTION
        }
    }

    override fun getMenuItemTitle(menuItem: SmartMenuItem): String = menuItem.label

    class EPGConfigValuesBackstage: EpgFragment.EPGConfigValues() {
        override val SCROLL_PAGE_SIZE = 5
        override val INACTIVITY_REFRESH_MS = 1000 * 60 * 30
        override val HORIZONTAL_PAGE_SIZE_MS = 1000 * 60 * 60 * 2
        override val VERTICAL_PAGE_SIZE_CHANNEL_COUNT = 8
        override val EPG_DAYS_INTO_PAST = 1
        override val EPG_DAYS_INTO_FUTURE = 1
        override val MILLISECONDS_IN_DAY = 24 * 60 * 60 * 1000
        override val REQUEST_FOCUS = "REQUEST_FOCUS"
    }
}